function Coeffs = Generate_Ts_Tm_coe(POS)
% POS is a array which contains all sites 'positions,
% POS first column stores all sites' longitudes,
% and second column must be all sites' latitudes.
% The number of input station positions is unlimited.
% Function return the time-varying global gridded Ts-Tm model function coefficients
% (Coeffs) at position POS

%%% Determined the total number of sites
[sitecount,~] = size(POS);

%%% Constants in function
% Radius Earth [m]; WGS-84
R_Earth = 6378.137e3;  

% Totol grid number in ERA-Interim's resolution
ERAI_NodeTotel = 512*256;

% Longtitude interval in ERA-Interim's resolution
ERAI_LonGridSpan = 360/512;

% Grid number along latitude line in ERA-Interim's resolution
ERAI_XNum = 512;

% Latitude series in N128 Gaussian grid
N128_Gaussian_Grid_Lat = [89.4628220000000,88.7669510000000,88.0669720000000,87.3660630000000,86.6648030000000,85.9633720000000,85.2618460000000,84.5602610000000,83.8586380000000,83.1569880000000,82.4553190000000,81.7536350000000,...
    81.0519400000000,80.3502370000000,79.6485270000000,78.9468110000000,78.2450910000000,77.5433670000000,76.8416400000000,76.1399100000000,75.4381770000000,74.7364430000000,74.0347070000000,73.3329700000000,...
    72.6312310000000,71.9294910000000,71.2277500000000,70.5260080000000,69.8242650000000,69.1225220000000,68.4207770000000,67.7190330000000,67.0172870000000,66.3155410000000,65.6137950000000,64.9120480000000,...
    64.2103010000000,63.5085540000000,62.8068060000000,62.1050580000000,61.4033090000000,60.7015600000000,59.9998110000000,59.2980620000000,58.5963130000000,57.8945630000000,57.1928140000000,56.4910640000000,...
    55.7893140000000,55.0875630000000,54.3858130000000,53.6840620000000,52.9823120000000,52.2805610000000,51.5788100000000,50.8770590000000,50.1753080000000,49.4735570000000,48.7718060000000,48.0700540000000,...
    47.3683030000000,46.6665510000000,45.9648000000000,45.2630480000000,44.5612960000000,43.8595450000000,43.1577930000000,42.4560410000000,41.7542890000000,41.0525370000000,40.3507850000000,39.6490330000000,...
    38.9472800000000,38.2455280000000,37.5437760000000,36.8420240000000,36.1402710000000,35.4385190000000,34.7367670000000,34.0350140000000,33.3332620000000,32.6315090000000,31.9297570000000,31.2280040000000,...
    30.5262520000000,29.8244990000000,29.1227460000000,28.4209940000000,27.7192410000000,27.0174880000000,26.3157360000000,25.6139830000000,24.9122300000000,24.2104770000000,23.5087250000000,22.8069720000000,...
    22.1052190000000,21.4034660000000,20.7017130000000,19.9999600000000,19.2982070000000,18.5964550000000,17.8947020000000,17.1929490000000,16.4911960000000,15.7894430000000,15.0876900000000,14.3859370000000,...
    13.6841840000000,12.9824310000000,12.2806780000000,11.5789250000000,10.8771720000000,10.1754190000000,9.47366600000000,8.77191300000000,8.07016000000000,7.36840700000000,6.66665400000000,5.96490100000000,...
    5.26314800000000,4.56139500000000,3.85964200000000,3.15788900000000,2.45613600000000,1.75438300000000,1.05263000000000,0.350877000000000,-0.350877000000000,-1.05263000000000,-1.75438300000000,-2.45613600000000,...
    -3.15788900000000,-3.85964200000000,-4.56139500000000,-5.26314800000000,-5.96490100000000,-6.66665400000000,-7.36840700000000,-8.07016000000000,-8.77191300000000,-9.47366600000000,-10.1754190000000,-10.8771720000000,...
    -11.5789250000000,-12.2806780000000,-12.9824310000000,-13.6841840000000,-14.3859370000000,-15.0876900000000,-15.7894430000000,-16.4911960000000,-17.1929490000000,-17.8947020000000,-18.5964550000000,-19.2982070000000,...
    -19.9999600000000,-20.7017130000000,-21.4034660000000,-22.1052190000000,-22.8069720000000,-23.5087250000000,-24.2104770000000,-24.9122300000000,-25.6139830000000,-26.3157360000000,-27.0174880000000,-27.7192410000000,...
    -28.4209940000000,-29.1227460000000,-29.8244990000000,-30.5262520000000,-31.2280040000000,-31.9297570000000,-32.6315090000000,-33.3332620000000,-34.0350140000000,-34.7367670000000,-35.4385190000000,-36.1402710000000,...
	  -36.8420240000000,-37.5437760000000,-38.2455280000000,-38.9472800000000,-39.6490330000000,-40.3507850000000,-41.0525370000000,-41.7542890000000,-42.4560410000000,-43.1577930000000,-43.8595450000000,-44.5612960000000,...
	  -45.2630480000000,-45.9648000000000,-46.6665510000000,-47.3683030000000,-48.0700540000000,-48.7718060000000,-49.4735570000000,-50.1753080000000,-50.8770590000000,-51.5788100000000,-52.2805610000000,-52.9823120000000,...
	  -53.6840620000000,-54.3858130000000,-55.0875630000000,-55.7893140000000,-56.4910640000000,-57.1928140000000,-57.8945630000000,-58.5963130000000,-59.2980620000000,-59.9998110000000,-60.7015600000000,-61.4033090000000,...
	  -62.1050580000000,-62.8068060000000,-63.5085540000000,-64.2103010000000,-64.9120480000000,-65.6137950000000,-66.3155410000000,-67.0172870000000,-67.7190330000000,-68.4207770000000,-69.1225220000000,-69.8242650000000,...
	  -70.5260080000000,-71.2277500000000,-71.9294910000000,-72.6312310000000,-73.3329700000000,-74.0347070000000,-74.7364430000000,-75.4381770000000,-76.1399100000000,-76.8416400000000,-77.5433670000000,-78.2450910000000,...
	  -78.9468110000000,-79.6485270000000,-80.3502370000000,-81.0519400000000,-81.7536350000000,-82.4553190000000,-83.1569880000000,-83.8586380000000,-84.5602610000000,-85.2618460000000,-85.9633720000000,-86.6648030000000,...
	  -87.3660630000000,-88.0669720000000,-88.7669510000000,-89.4628220000000];

% four neighboring grids' serial number
NodeID(sitecount,4) = 0;
% four neighboring grids' interpolation coefficients
NodeInterpC(sitecount,4) = 0;

% At each given site, determine the interpolation coefficients of each neighboring grid to site's position
for sc = 1 : sitecount
    lon = POS(sc,1);
    lat = POS(sc,2);
    
    if lon < 0
        lon = lon + 360;
    end;
    
    % Latitude > 89.462822
    if lat > N128_Gaussian_Grid_Lat(1) 
        NodeID(sc,:) = 1;
        NodeInterpC(sc,1) = 1;
    % Latitude < -89.462822  
    elseif lat < N128_Gaussian_Grid_Lat(256) 
        NodeID(sc,:) = ERAI_NodeTotel;
        NodeInterpC(sc,1) = 1;   
    % -89.462822 < Latitude < 89.462822
    else
        iLon_grid = floor(lon/ERAI_LonGridSpan);
        if iLon_grid > 511
            iLon_grid = 511;
        end;
        
        iLat_grid = 1;
        while N128_Gaussian_Grid_Lat(iLat_grid) > lat
            iLat_grid = iLat_grid + 1;
        end;
        
        cosB = cosd(lat);
        sinB = sind(lat);
        
        %  p3   p4
        %  p1   p2       
        cosPsi = sind(N128_Gaussian_Grid_Lat(iLat_grid-1))*sinB...
            + cosd(N128_Gaussian_Grid_Lat(iLat_grid-1))*cosB*cosd(ERAI_LonGridSpan * (iLon_grid+1) - lon);
        psi = acos(cosPsi);
        w4 = 1 / (R_Earth * psi);
        
        cosPsi = sind(N128_Gaussian_Grid_Lat(iLat_grid-1))*sinB...
            + cosd(N128_Gaussian_Grid_Lat(iLat_grid-1))*cosB*cosd(ERAI_LonGridSpan * iLon_grid - lon);
        psi = acos(cosPsi);
        w3 = 1 / (R_Earth * psi);
        
        cosPsi = sind(N128_Gaussian_Grid_Lat(iLat_grid))*sinB...
            + cosd(N128_Gaussian_Grid_Lat(iLat_grid))*cosB*cosd(ERAI_LonGridSpan * (iLon_grid+1) - lon);
        psi = acos(cosPsi);
        w2 = 1 / (R_Earth * psi);
        
        cosPsi = sind(N128_Gaussian_Grid_Lat(iLat_grid))*sinB...
            + cosd(N128_Gaussian_Grid_Lat(iLat_grid))*cosB*cosd(ERAI_LonGridSpan * iLon_grid - lon);
        psi = acos(cosPsi);
        w1 = 1 / (R_Earth * psi);
        
        w = w1 + w2 + w3 + w4;
        
        NodeID(sc,1) = (iLat_grid-1) * ERAI_XNum + iLon_grid+1;
        NodeInterpC(sc,1) = w1 / w;
        
        NodeID(sc,3) = (iLat_grid-2) * ERAI_XNum + iLon_grid+1;
        NodeInterpC(sc,3) = w3 / w;
        
        if 511 == iLon_grid 
            NodeID(sc,2) = (iLat_grid-1)*ERAI_XNum+1;
            NodeInterpC(sc,2) = w2/w;
            
            NodeID(sc,4) = (iLat_grid-2)*ERAI_XNum+1;
            NodeInterpC(sc,4) = w4/w;
        else
            NodeID(sc,2) = (iLat_grid-1)*ERAI_XNum+iLon_grid+2;
            NodeInterpC(sc,2) = w2/w;
            
            NodeID(sc,4) = (iLat_grid-2)*ERAI_XNum+iLon_grid+2;
            NodeInterpC(sc,4) = w4/w;
        end;
        
        clear w1 w2 w3 w4 w cosPsi psi cosB sinB iLat_grid iLon_grid;        
    end;    
    clear lon lat;
end;
clear  N128_Gaussian_Grid_Lat;


%%% Generate coefficients in Time-Varying Global Gridded (TVGG) model at given position
%%% by horizontally interpolating coefficients of neighboring ERA-Interim grids 
modelfun=@(f,X)f(1)*X(:,1)+f(2)+f(3)*cos(X(:,2)*2*pi/365.25)+f(4)*sin(X(:,2)*2*pi/365.25)+f(5)*cos(X(:,2)*4*pi/365.25)+f(6)*sin(X(:,2)*4*pi/365.25)+f(7)*cos(X(:,3)*2*pi/24)+f(8)*sin(X(:,3)*2*pi/24);

% Get function coefficients of all 512*256 ERA-Interim grids
Coeff_TVGG_ERAI = load('Coeff_TVGG_ERAI.mat');

% Interpolate function coefficients to sites' positions 
for sc = 1 : sitecount
    C_temp = Coeff_TVGG_ERAI.Coeff_TVGG_ERAI(NodeID(sc,:),:);    
    Coeffs(:,sc) = C_temp'*NodeInterpC(sc,:)';
    clear C_temp;
end;

Coeffs = Coeffs';

