#Supplement to Andrews et al., "CO2, CO and CH4 Measurements from the 
#NOAA Earth System Research Laboratory’s Tall Tower Greenhouse Gas Observing Network: 
#Instrumentation, Uncertainty Analysis and Recommendations for Future High-Accuracy 
#Greenhouse Gas Monitoring Efforts", Atmospheric Measurement Techniques, under review.

#Uncertainty Equations and Monte Carlo simulation described in Appendix D and 
#used to generate Figure D1.

#number of iterations for the monte carlo values
niter = 2500

#user defined x values representing calibration standards with known value
x = c(350,380,410,460)
#analyzer baseline and slope
b = -370
m = 0.8
#noise free analyzer signals
y = m*x + b


pdfout = "./monte_carlo_2013_10_09.pdf"  
#pdfout = NULL
#set pdfout = NULL if screen plots are desired (if running interactively within R)
#Direct plot output to file or prompt for next plot if running interactively:
if(!is.null(pdfout)){ pdf(file=pdfout) } else { par(ask=T) } 

#output directories for textfiles (must exist prior to execution)
txtoutdir = "./monte_carlo_out/"  
#txtoutdir = NULL
#set txtoutdir = NULL if no text output is desired

#define two cases with differing x uncertainty and identical fit residuals, Equation (D3)
xuncvec = c(0.03, 1e-10)
sigmay = 0.05

for(j in 1:length(xuncvec)){
	xunc = xuncvec[j]

	# Compute the portion of sigmay caused by analytical system, Equation (3)
	if(sigmay < m*xunc) stop("sigmay must be > m*xunc")
	yunc = sqrt(sigmay^2-(m*xunc)^2) 

	#define a vector containing a range of x values to simulate "uknown" samples
	#then compute corresponding y values 
	xtest<-seq(min(x)-abs(diff(range(x))),max(x)+abs(diff(range(x))),length=20)
	ytest = m*xtest + b 

	#define empty matrices and vectors to hold monte carlo output 
	coefmat = matrix(NA, ncol=2,nrow=niter)
	dimnames(coefmat) = list(NULL, c("b","m"))
	coefuncmat = matrix(NA, ncol=3,nrow=niter)
	dimnames(coefuncmat) = list(NULL, c("b_unc","m_unc","b_unc_min"))
	xsammat = matrix(NA, ncol=length(ytest),nrow=niter)
	sigma = rep(NA, length=niter)

	#loop over n iterations for the monte carlo
	for(i in 1:niter){
	  #use "rnorm" function to add noise to x and y values
	  #rnorm(n) generates a vector of random numbers drawn from a distribution with 
	  #zero mean and std dev = 1
	  #multiply by "xunc" and "yunc" to scale appropriately
	  xi = x+xunc*rnorm(length(x))
	  yi = y+yunc*rnorm(length(x))
  
	  #the next few lines take advantage of built-in regression functions in R to compute 
	  #regression coefficient uncertainties
	  #this could alternatively be done using Equations D2 - D5
	  #lm is "linear model", which here computes a 1st order regression with an intercept
	  tmpfit = lm(yi~xi)
	  #remove the mean to obtain minium value of the interecept and intercept uncertainty
	  xi2 = xi - mean(xi)
	  tmpfit2 = lm(yi~xi2)
  
	  #extract intercept and intercept uncdertainty
	  bi = tmpfit$coef[1]
	  biunc = summary(tmpfit)$coef[1,"Std. Error"]
	  biuncmin = summary(tmpfit2)$coef[1,"Std. Error"]
	  #extract slope and slpe umcertainty
	  mi = tmpfit$coef[2]
	  miunc = summary(tmpfit)$coef[2,"Std. Error"]
	  #extract std deviation of the residuals (i.e. sigmay for this iteration)
	  sigmayi = summary(tmpfit)$sigma
  
	  #generate a vector of analyzer signals with noise corresponding to unknown 
	  #samples & comptue corresponding x values
	  ysam = ytest + yunc*rnorm(length(ytest))
	  xsam = (ysam-bi)/mi
  
	  #store results and clean up
	  coefmat[i,] = c(bi,mi)
	  coefuncmat[i,] = c(biunc,miunc,biuncmin)
	  sigma[i] = sigmayi
	  xsammat[i,] = xsam
	  rm(xsam,ysam, xi,yi,tmpfit,mi,bi,miunc,biunc,biuncmin,sigmayi)

	} #end for i in 1:niter

	#This is the standard deviation across niter realizations of the experiment
	# i.e. the R apply function computes the standard deviation across the rows of xsammat
	SDu.monte = apply(xsammat, MARGIN=2, FUN=sd)

	#For lines on plots with finer resolution and covering a wide range of x values
	xtestx = seq(from=200, to = 650, by =1)
	ytestx = m*xtestx + b 


	#This is the Skoog and Leary equation for the predicted standard deviation of the 
	#unknown samples, Equation D6
	SDu.skoog = (sigmay/m)*sqrt(1+(1/length(x))+((ytestx-mean(y))^2)/(m^2*sum((x-mean(x))^2)))

	#std dev of slope: Equation D2
	sigmam<-sigmay/sqrt(sum((x-mean(x))^2))
	
	#std dev of intercept: Equation D4
	#shift x values by subtracting mean(x) to get stdbmin
	sigmabmin<- sigmay*sqrt(sum((x-mean(x))^2)/(length(x)*sum((x-mean(x))^2) - sum((x-mean(x)))^2))

	#standard error of the fit: Equation D5
	sefit = sqrt((sigmam*(xtestx-mean(x)))^2+sigmabmin^2)

	#Equation D10
	SDu.alt = sqrt((sefit/m)^2 + (yunc/m)^2)

	#create a plot with title showing summary of input parameters
	plot(xtest, SDu.monte, ylim=range(c(0,SDu.monte, SDu.skoog, SDu.alt)),xlab="x values", ylab="Standard Deviation, x units", cex.lab=1.5, cex.axis=1.5)
	lines(xtestx, SDu.skoog, col=3, lwd=2)
	lines(xtestx, SDu.alt, col=1, lty=2, lwd=2)
	mcinfo<-paste("sigmay=",round(sigmay,2),";yunc=",round(yunc,2),";xunc=",round(xunc,2),";m=",m,";b=",b,";niter=",niter, sep="")
	title(mcinfo)

	#write output to text files
	if(!is.null(txtoutdir)){
	  write.table(coefmat,file=paste(txtoutdir,"coefs",j,".out",sep=""),row.names=F, col.names=T, na="-999") 
	  write.table(coefuncmat,file=paste(txtoutdir,"coefunc",j,".out",sep=""),row.names=F, col.names=T, na="-999") 
	  write(sigma,file=paste(txtoutdir,"sigma",j,".out",sep=""),ncol=1) 
	  write.table(xsammat,file=paste(txtoutdir,"xsam",j,".out",sep=""),row.names=F, col.names=F, na="-999") 
	  write(ytest,file=paste(txtoutdir,"ytest",j,".out",sep=""),ncol=1) 
	  write.table(data.frame(x=x,y=y),file=paste(txtoutdir,"xy",j,".out",sep=""),row.names=F, col.names=T, na="-999")
	  write(mcinfo,  file=paste(txtoutdir,"mcinfo",j,".out",sep=""))
	}
 
} #loop over xunc


if(!is.null(pdfout)) dev.off()


