## R program 'Figure_Timeseries_Power_Rainfall.R'.
## Program for rainfall estimation using microwave links.
## Make plots of time series of received powers, rainfall intensities, et cetera.
## Described in paper:
## Aart Overeem, Hidde Leijnse, Remko Uijlenhoet, 2015. Retrieval algorithm for rainfall mapping from microwave links in a cellular communication network. Atmos. Meas. Tech. Discuss., submitted.  

## Copyright (C) 2015 Aart Overeem
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program. If not, see <http://www.gnu.org/licenses/>.


# source('Figure_Timeseries_Power_Rainfall.R')
# Start R and go to the directory "WorkingExample".  Then run this script by pasting the line above (source(...)).
#rm(list=ls(all=TRUE))  
startTime = proc.time()	

# Script has to be adapted in case of time interval other than 15 min. E.g., the construction of the scale needs to be modified.
# Scale of vertical axis needs to be changed in case of power signal levels or rainfall values which clearly differ from this example.

# Create directory for output files:
dir.create("Figures")

# Make list of complete date objects:
Date_complete = strptime("0815", "%H%M",tz="UTC")
for (i in 1:95)
{
	Date_complete[i+1] = Date_complete[1] + i * 900
}

# Choose for which period the time series should be plotted. Here: time interval 32 up to and including time interval 68.
T1 = 40
T2 = 64

# Read link data
DATE = 20110911
u = read.table(paste("LinkData_Corrected/linkdata_15min",DATE,"_val.dat",sep=""),header=TRUE)


# Read radar data
w = read.table("radar_gaugeadjusted_15min.dat")
rad_mean = w[,2] * 4  # Convert from 15-minrainfall depth (mm) to rainfall intensity (mm/h).


# Select data from link with identifier 2602:
cond = which(u$ID==2602)

# Obtain date object and determine end time of last time step:
Index=c(NA)
Index = strptime(u$DATETIME, "%Y%m%d%H%M",tz="UTC")
DateTimeEnd = max(Index)
# For the selected link no filter has to be applied:
# cond2 = which(u$ID==unique(u$ID[cond]))
# min(u$F[cond2])
# -13.28061  # This is much larger than the threshold value of -32.5 dB/km h. So no filter has to be applied for this link. 
# Note that the filter is NOT implemented in this script.


# Range vertical axis for powers:
ylimP = c(-100,-30)
# Range vertical axis for minimum and maximum rainfall intensity:
ylimRminmax = c(0,110)
# Range vertical axis for mean rainfall intensity:
ylimRmean = c(0,40)
# Range vertical axis for cumulative rainfall depth:
ylimRcum = c(0,30)



# MAKE TIME SERIES GRAPHS

# Graph 1:
filename=paste("Figures/PLinkUncor_RintmeanRadarsGauges_",DATE,".pdf",sep="")
pdf(filename,family="Times")
par(pty="s")  
par(ps=24)
par(mar=c(1,5.5,2,4.5))
par(mfrow=c(1,1))
par(cex.axis=1)  

# Plot minimum received power signals
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = u$RXMIN[cond]
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimP, axes=TRUE,type="l",col="darkblue",cex=.2, lwd=4,ylab="",xlab="",lty=1)
mtext("Received power (dBm)",side=2,col="black",line=3,cex=1,cex.axis=1)	
mtext("Time (UTC)",side=1,col="black",line=3,cex=1) 	      
	   
par(ps=20)
# Plot link length, microwave frequency, and time resolution
linklength = u$path_length[cond]
linklength = formatC(unique(linklength[!is.na(linklength)]), format="f", digits=1)
freq = u$frequency[cond]
freq = formatC(unique(freq[!is.na(freq)]), format="f", digits=1)
legend("topright",c(substitute(italic(f) == freq*" GHz",list(freq=freq)),substitute(italic(L) == linklength*" km",list(linklength=linklength)),expression(Delta * italic(t)*" = 15 min")),bty="n",cex=0.9,y.intersp=1.15)
par(ps=24)
	   
# Plot mean radar rainfall intensity over the link
par(new=TRUE)
plot(Date_complete[T1:T2], rad_mean[T1:T2], axes=FALSE,ylab="",xlab="",col="gray55",type="l",ylim=ylimRmean,lwd=3,lty=1)
axis(4, ylim=ylimRmean,col="black")
mtext(expression("Mean radar rainfall intensity (mm h"^-1*")"),side=4,line=3,cex=1,cex.axis=1) 
	      	   
# Plot maximum received power signals
par(new=TRUE)
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = u$RXMAX[cond]
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimP,axes=TRUE,type="l",col="black",
cex=.5,lwd=4,ylab="",xlab="",cex.main=0.7,main=paste("Link approach, filter. End: ",DateTimeEnd,sep=""))
	
# Plot legend   
par(ps=20)	   
legend("topleft",legend=c(expression(italic(P)[max]),expression(italic(P)[min]),
"Radars + gauges (link path)"),text.col=c("black","darkblue","gray55"),col=c("black","darkblue","gray55"),bty="n",lwd=c(4,4,3),lty=c(1,1,1),cex=0.9,y.intersp=1.15)     
par(ps=24)	  

dev.off() 



# Graph 2:
filename=paste("Figures/PLinkCor_RintmeanRadarsGauges_",DATE,".pdf",sep="")
pdf(filename,family="Times")
par(pty="s")  
par(ps=24)
par(mar=c(1,5.5,2,4.5))
par(mfrow=c(1,1))
par(cex.axis=1)   

# Plot minimum received power signals
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = u$RXMIN_final[cond]
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimP, axes=TRUE,type="l",col="darkblue",cex=.2, lwd=4,ylab="",xlab="",lty=1)
mtext("Corrected received power (dBm)",side=2,col="black",line=3,cex=1,cex.axis=1)	
mtext("Time (UTC)",side=1,col="black",line=3,cex=1) 	      
axis(2, ylim=ylimP,col="black") 
	   
par(ps=20)
# Plot link length, microwave frequency, and time resolution
linklength = u$path_length[cond]
linklength = formatC(unique(linklength[!is.na(linklength)]), format="f", digits=1)
freq = u$frequency[cond]
freq = formatC(unique(freq[!is.na(freq)]), format="f", digits=1)
legend("topright",c(substitute(italic(f) == freq*" GHz",list(freq=freq)),substitute(italic(L) == linklength*" km",list(linklength=linklength)),expression(Delta * italic(t)*" = 15 min")),bty="n",cex=0.9,y.intersp=1.15)
par(ps=24)
	   
# Plot mean radar rainfall intensity over the link
par(new=TRUE)
plot(Date_complete[T1:T2], rad_mean[T1:T2], axes=FALSE,ylab="",xlab="",col="gray55",type="l",ylim=ylimRmean,lwd=3,lty=1)
axis(4, ylim=ylimRmean,col="black")
mtext(expression("Mean radar rainfall intensity (mm h"^-1*")"),side=4,line=3,cex=1,cex.axis=1) 
	      	   
# Plot maximum received power signals
par(new=TRUE)
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = u$RXMAX_final[cond]
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimP,axes=TRUE,type="l",col="black",
cex=.5,lwd=4,ylab="",xlab="",cex.main=0.7,main=paste("Link approach, filter. End: ",DateTimeEnd,sep=""))

# Plot reference level
par(new=TRUE)
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = u$ref_level[cond]
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimP,axes=TRUE,type="l",col="orange",cex=.5,lwd=4,ylab="",xlab="",cex.main=0.7)
	
# Plot legend   
par(ps=20)	   
legend("topleft",legend=c(expression(italic(P)[ref]),expression(italic(P)[max]^C),expression(italic(P)[min]^C),"Radars + gauges (link path)"),text.col=c("orange","black","darkblue","gray55"),
col=c("orange","black","darkblue","gray55"),bty="n",lwd=c(4,4,4,3),lty=c(1,1,1,1),cex=0.9,y.intersp=1.15)     
par(ps=24)	  

dev.off() 



# Graph 3:
filename=paste("Figures/RintminRintmaxLink_",DATE,".pdf",sep="")
pdf(filename,family="Times")
par(pty="s")  
par(ps=24)
par(mar=c(1,5.5,2,4.5))
par(mfrow=c(1,1))
par(cex.axis=1)   

Aa = 2.3
gsm_min = gsm_max = c(NA)
for (s in 1:length(cond))
{
	gsm_max[s] = max(u$a[cond][s]*( (u$Am_max[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
	gsm_min[s] = max(u$a[cond][s]*( (u$Am_min[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
}  
 
# Plot minimum link rainfall intensity
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = gsm_min
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimRminmax,type="l",col="gray55",cex=.2, lwd=4,ylab="",xlab="",lty=1)
mtext(expression("Link rainfall intensity (mm h"^-1*")"),side=2,col="black",line=3,cex=1,cex.axis=1)	
mtext("Time (UTC)",side=1,col="black",line=3,cex=1) 	      
axis(2, ylim=ylimRminmax,col="black") 
	   
par(ps=20)
# Plot link length, microwave frequency, and time resolution
linklength = u$path_length[cond]
linklength = formatC(unique(linklength[!is.na(linklength)]), format="f", digits=1)
freq = u$frequency[cond]
freq = formatC(unique(freq[!is.na(freq)]), format="f", digits=1)
legend("topright",c(substitute(italic(f) == freq*" GHz",list(freq=freq)),substitute(italic(L) == linklength*" km",list(linklength=linklength)),expression(Delta * italic(t)*" = 15 min")),bty="n",cex=0.9,y.intersp=1.15)
par(ps=24)
	      	   
# Plot maximum link rainfall intensity
par(new=TRUE)
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = gsm_max
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimRminmax,type="l",col="black",
cex=.5,lwd=4,ylab="",xlab="",cex.main=0.7,main=paste("Link approach, filter. End: ",DateTimeEnd,sep=""))
	
# Plot legend   
par(ps=20)	   
legend("topleft",legend=c("Max","Min"),text.col=c("black","gray55"),
col=c("black","gray55"),bty="n",lwd=c(4,4),lty=c(1,1),cex=0.9,y.intersp=1.15)     
par(ps=24)	  

dev.off() 



# Graph 4:
filename=paste("Figures/RintmeanLink_RintmeanRadarsGauges_",DATE,".pdf",sep="")
pdf(filename,family="Times")
par(pty="s")  
par(ps=24)
par(mar=c(1,5.5,2,4.5))
par(mfrow=c(1,1))
par(cex.axis=1)   

Aa = 2.3
gsm_min = gsm_max = c(NA)
for (s in 1:length(cond))
{
	gsm_max[s] = max(u$a[cond][s]*( (u$Am_max[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
	gsm_min[s] = max(u$a[cond][s]*( (u$Am_min[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
}  
alpha = 0.33
Rgsm = c(NA)
Rgsm = (1 - alpha) * gsm_min + alpha * gsm_max 
 

# Plot mean link rainfall intensity
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = Rgsm
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimRmean,type="l",col="black",cex=.2, lwd=4,ylab="",xlab="",lty=1)
mtext(expression("Mean rainfall intensity (mm h"^-1*")"),side=2,col="black",line=3,cex=1,cex.axis=1)	
mtext("Time (UTC)",side=1,col="black",line=3,cex=1) 	      
axis(2, ylim=ylimRmean,col="black") 
	   
par(ps=20)
# Plot link length, microwave frequency, and time resolution
linklength = u$path_length[cond]
linklength = formatC(unique(linklength[!is.na(linklength)]), format="f", digits=1)
freq = u$frequency[cond]
freq = formatC(unique(freq[!is.na(freq)]), format="f", digits=1)
legend("topright",c(substitute(italic(f) == freq*" GHz",list(freq=freq)),substitute(italic(L) == linklength*" km",list(linklength=linklength)),expression(Delta * italic(t)*" = 15 min")),bty="n",cex=0.9,y.intersp=1.15)
par(ps=24)
	      	   
# Plot mean radar rainfall intensity over the link
par(new=TRUE)
plot(Date_complete[T1:T2], rad_mean[T1:T2],ylab="",xlab="",col="gray55",type="l",ylim=ylimRmean,lwd=3,lty=1,cex.main=0.7,main=paste("Link approach, filter. End: ",DateTimeEnd,sep=""))
	
# Plot legend   
par(ps=20)	   
legend("topleft",legend=c("Link","Radars + gauges (link path)"),text.col=c("black","gray55"),
col=c("black","gray55"),bty="n",lwd=c(4,3),lty=c(1,1),cex=0.9,y.intersp=1.15)     
par(ps=24)	  

dev.off() 



# Graph 5:
filename=paste("Figures/RcumLink_RcumRadarsGauges_",DATE,".pdf",sep="")
pdf(filename,family="Times")
par(pty="s")  
par(ps=24)
par(mar=c(1,5.5,2,4.5))
par(mfrow=c(1,1))
par(cex.axis=1)   

Aa = 2.3
gsm_min = gsm_max = c(NA)
for (s in 1:length(cond))
{
	gsm_max[s] = max(u$a[cond][s]*( (u$Am_max[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
	gsm_min[s] = max(u$a[cond][s]*( (u$Am_min[cond][s]-Aa) /u$path_length[cond][s])^u$b[cond][s],0,na.rm=T)
}  
alpha = 0.33
Rgsm = c(NA)
Rgsm = (1 - alpha) * gsm_min + alpha * gsm_max 
 

# Plot cumulative link rainfall depth
temp = c(NA)
temp[1:96] = NA
temp[u$interval_number[cond]] = cumsum(Rgsm)/4
plot(Date_complete[T1:T2], temp[T1:T2], ylim=ylimRcum,type="l",col="black",cex=.2, lwd=4,ylab="",xlab="",lty=1)
mtext("Cumulative rainfall depth (mm)",side=2,col="black",line=3,cex=1,cex.axis=1)	
mtext("Time (UTC)",side=1,col="black",line=3,cex=1) 	      
axis(2, ylim=ylimRcum,col="black") 
	   
par(ps=20)
# Plot link length, microwave frequency, and time resolution
linklength = u$path_length[cond]
linklength = formatC(unique(linklength[!is.na(linklength)]), format="f", digits=1)
freq = u$frequency[cond]
freq = formatC(unique(freq[!is.na(freq)]), format="f", digits=1)
legend("bottomright",c(substitute(italic(f) == freq*" GHz",list(freq=freq)),substitute(italic(L) == linklength*" km",list(linklength=linklength)),expression(Delta * italic(t)*" = 15 min")),bty="n",cex=0.9,y.intersp=1.15)
par(ps=24)
	      	   
# Plot cumulative radar rainfall depth
par(new=TRUE)
CumSum = cumsum(rad_mean)/4
plot(Date_complete[T1:T2], CumSum[T1:T2], ylab="",xlab="",col="gray55",type="l",ylim=ylimRcum,lwd=3,lty=1,cex.main=0.7,main=paste("Link approach, filter. End: ",DateTimeEnd,sep=""))
	
# Plot legend   
par(ps=20)	   
legend("topleft",legend=c("Link","Radars + gauges (link path)"),text.col=c("black","gray55"),
col=c("black","gray55"),bty="n",lwd=c(4,3),lty=c(1,1),cex=0.9,y.intersp=1.15)     
par(ps=24)	  

dev.off()    



# MAKE MAP WITH LOCATION OF LINK
#pathlib = "path name"
library(ggplot2)	#,lib.loc=pathlib)
library(ggmap)
library(maps)
library(mapproj)
library(labeling)
# OpenStreetMap (street map)
data_coor = read.table("WGS84_coordinates_Utrecht_20110911_LinkTimeseries.dat")
map <- get_openstreetmap(bbox = c(left = 5.09, bottom = 52.07, right = 5.13, top = 52.09),format="png",scale=7500) 	
tiff("Figures/Utrecht_linknetwork_20110911_LinkTimeseries_OSM.tiff", width = 2332, height = 1897.5, compression = c("none"), units = "px")   	# height: 1725 *1.1 = 1897.5; width = 2120*1.1 = 2332
par(family="Times")
plotmap = ggmap(map) + geom_segment(aes(x=data_coor[,1],y=data_coor[,2],xend=data_coor[,3],yend=data_coor[,4]),data=data_coor,alpha=0.5,col="blue",size=20) + theme(axis.title.x=element_text(size =rel(10),family="Times")) + xlab("Longitude (º)") + theme(axis.title.y=element_text(size =rel(10),family="Times")) + ylab("Latitude (º)") + theme(axis.text = element_text(size=rel(7),family="Times")) + theme(axis.ticks = element_line(size = 22))
print(plotmap)
dev.off()

cat(sprintf("Finished. (%.1f seconds)\n",round((proc.time()-startTime)[3],digits=1)))

