## R program 'RainfallMapping.R'.
## Program for rainfall estimation using microwave links.
## Visualisation: 15-min and daily rainfall maps.
## Described in paper:
## Aart Overeem, Hidde Leijnse, Remko Uijlenhoet, 2015. Retrieval algorithm for rainfall mapping from microwave links in a cellular communication network. Atmos. Meas. Tech. Discuss., submitted.  

## Copyright (C) 2015 Aart Overeem
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program. If not, see <http://www.gnu.org/licenses/>.


# Note that a local Cartesian coordinate system is used.
# source("RainfallMapping.R")
# Start R and go to the directory "WorkingExample". Then run this script by pasting the line above (source(...)).
rm(list=ls(all=TRUE))
startTime = proc.time()             # Get starting time


# Load R libraries:
#pathlib = "path name"
library(labeling)	#,lib.loc=pathlib)
library(ggplot2)
require(grid)


# Read interpolation grid (should be supplied in WGS84 in degrees):
# Load R library
library(sp)
library(rgdal)
path.grid = "InterpolationGrid.dat"
rain.grid = read.table(path.grid,header=TRUE,sep=",")
# X_middle = center of coordinate system (longitude); Y_middle = center of coordinate system (latitude); 
# We chose 52.155223°N 5.387242°E as the middle of the Netherlands ('The Tower of Our Lady' is a church tower in Amersfoort and used to be the middle point of the Dutch grid reference system).
X_middle = 5.387242
Y_middle = 52.155223
# Convert WGS84 coordinates (degrees) to an Azimuthal Equidistant Cartesian coordinate system (easting and northing of grid; m).
# of start of link, easting and northing of end of link, respectively; km). 
d <- data.frame(lon=rain.grid$X,lat=rain.grid$Y)
coordinates(d) <- c("lon", "lat")
proj4string(d) <- CRS("+init=epsg:4326") # WGS 84
CRS.new <- CRS(paste("+proj=aeqd +a=6378.137 +b=6356.752 +R_A +lat_0=",Y_middle," +lon_0=",X_middle," +x_0=0 +y_0=0",sep=""))
UTM <- spTransform(d, CRS.new)
#UTM$lon  Easting (in km)
#UTM$lat  Northing (in km)
rain.grid = data.frame(cbind(UTM$lon,UTM$lat))



# Create directory for output files:
dir.create("Figures")



#######################
# 15-MIN RAINFALL MAPS#
#######################
# Rainfall threshold: values below this threshold are set to NA.
threshold = 0.1


# Used time step in minutes:
Timestep = 15

MinutesInHour = 60
ConversionRainfallIntensity_To_RainfallDepth = Timestep/MinutesInHour


# Make list of input files: 
files_link <- list.files(path = paste("RainfallMaps_15min_links"), all.files=FALSE,full.names=TRUE, recursive=FALSE, pattern="kriging_20110911")
files_link = files_link[which(file.info(files_link)$size>0)]
if (length(files_link)==0)
{
	print("No files with data! Program stops.")
	stop()
}


# Define minimum and maximum rainfall accumulation (mm) to be plotted:
min.R = 0
max.R = 13


i = 47
# Read rainfall intensity from kriged map, convert to rainfall depth (mm):
rain.links = read.table(files_link[i],header=TRUE) * ConversionRainfallIntensity_To_RainfallDepth
# Replace values below threshold by NA:
rain.links[which(rain.links$rain_intensity<threshold),1] = NA

x = rain.grid[,1]
y = rain.grid[,2]
z = rain.links$rain_intensity
dataf = data.frame(cbind(x,y,z))

# Plot link rainfall map:
pdf("Figures/Link_15min_RainfallDepth_201109102045.pdf",family="Times")
fig = ggplot(dataf, aes(x = x, y = y, color = z)) + geom_point(shape=15,size=0.55) + labs(x = "Easting (km)", y = "Northing (km)", colour = "15-min rainfall depth (mm)",size=18) + coord_fixed() + ggtitle("Links only, link approach, filter") + theme(legend.position=c(0.185,0.87),legend.background = element_rect(fill="white"),panel.grid.major = element_blank(),panel.grid.minor = element_blank(),legend.key.width = unit(2, "cm"),legend.title = element_text(size = 10, colour = "black"),legend.text = element_text(size = 13, colour = "black")) + scale_colour_gradientn(colours = rainbow(7),na.value = "gray",limits=c(min.R,max.R)) + theme(plot.title=element_text(family="Times", face="bold", size=18),axis.title=element_text(family="Times", face="bold", size=18),axis.text=element_text(family="Times", face="bold", size=18))
print(fig)
dev.off()


# Read rainfall depth (mm) from gauge-adjusted radar data file:
rain.radar = read.table("Radar_201109102045_15min.dat",header=TRUE) 
# Replace values below threshold by NA:
rain.radar[which(rain.radar[,1]<threshold),1] = NA

x = rain.grid[,1]
y = rain.grid[,2]
z = rain.radar[,1]
dataf = data.frame(cbind(x,y,z))
# Maximum rainfall depth:
max(dataf[,3],na.rm=T)

# Plot gauge-adjusted radar rainfall map:
pdf("Figures/Radar_15min_RainfallDepth_201109102045.pdf",family="Times")
fig = ggplot(dataf, aes(x = x, y = y, color = z)) + geom_point(shape=15,size=0.55) + labs(x = "Easting (km)", y = "Northing (km)", colour = "15-min rainfall depth (mm)",size=18) + coord_fixed() + ggtitle("Gauge-adjusted radar data") + theme(legend.position=c(0.185,0.87),legend.background = element_rect(fill="white"),panel.grid.major = element_blank(),panel.grid.minor = element_blank(),legend.key.width = unit(2, "cm"),legend.title = element_text(size = 10, colour = "black"),legend.text = element_text(size = 13, colour = "black")) + scale_colour_gradientn(colours = rainbow(7),na.value = "gray",limits=c(min.R,max.R)) + theme(plot.title=element_text(family="Times", face="bold", size=18),axis.title=element_text(family="Times", face="bold", size=18),axis.text=element_text(family="Times", face="bold", size=18))
print(fig)
dev.off()



######################
# DAILY RAINFALL MAPS#
######################

# Compute daily rainfall depths from mean 15-min rainfall intensities:
files_link <- list.files(path = paste("RainfallMaps_15min_links"), all.files=FALSE,full.names=TRUE, recursive=FALSE, pattern="kriging_20110911")
files_link = files_link[which(file.info(files_link)$size>0)]
if (length(files_link)==0)
{
	print("No files with data! Program stops.")
	stop()
}
R_day = c(NA)
NrGridPoints = 38063
R_day[1:NrGridPoints] = 0
test = array(NA,c(NrGridPoints,length(files_link)))
for (z in 1:length(files_link))
{
	u = read.table(files_link[z],header=TRUE)
	R_day[1:NrGridPoints] = R_day[1:NrGridPoints] + u$rain_intensity[1:NrGridPoints]
}
R_day = R_day*ConversionRainfallIntensity_To_RainfallDepth


# Rainfall threshold: values below this threshold are set to NA.
threshold = 1

# Define minimum and maximum rainfall accumulation (mm) to be plotted:
min.R = 0
max.R = 50


# Make data frame for plotting:
x = rain.grid[,1]
y = rain.grid[,2]
z = R_day
z[which(z<threshold)] = NA
dataf = data.frame(cbind(x,y,z))

# Plot link rainfall map:
pdf("Figures/Link_Daily_RainfallDepth_20110911.pdf",family="Times")
fig = ggplot(dataf, aes(x = x, y = y, color = z)) + geom_point(shape=15,size=0.55) + labs(x = "Easting (km)", y = "Northing (km)", colour = "Daily rainfall depth (mm)",size=18) + coord_fixed() + ggtitle("Links only, link approach, filter") + theme(legend.position=c(0.185,0.87),legend.background = element_rect(fill="white"),panel.grid.major = element_blank(),panel.grid.minor = element_blank(),legend.key.width = unit(2, "cm"),legend.title = element_text(size = 10, colour = "black"),legend.text = element_text(size = 13, colour = "black")) + scale_colour_gradientn(colours = rainbow(7),na.value = "gray",limits=c(min.R,max.R)) + theme(plot.title=element_text(family="Times", face="bold", size=18),axis.title=element_text(family="Times", face="bold", size=18),axis.text=element_text(family="Times", face="bold", size=18))
print(fig)
dev.off()



# Read daily gauge-adjusted radar rainfall depths:
radar = read.table("Radar_20110911_daily.dat",header=TRUE)[,1]

# Make data frame for plotting:
x = rain.grid[,1]
y = rain.grid[,2]
z = radar
z[which(z<threshold)] = NA
dataf = data.frame(cbind(x,y,z))

# Plot gauge-adjusted radar rainfall map:
pdf("Figures/Radar_Daily_RainfallDepth_20110911.pdf",family="Times")
fig = ggplot(dataf, aes(x = x, y = y, color = z)) + geom_point(shape=15,size=0.55) + labs(x = "Easting (km)", y = "Northing (km)", colour = "Daily rainfall depth (mm)",size=18) + coord_fixed() + ggtitle("Gauge-adjusted radar data") + theme(legend.position=c(0.185,0.87),legend.background = element_rect(fill="white"),panel.grid.major = element_blank(),panel.grid.minor = element_blank(),legend.key.width = unit(2, "cm"),legend.title = element_text(size = 10, colour = "black"),legend.text = element_text(size = 13, colour = "black")) + scale_colour_gradientn(colours = rainbow(7),na.value = "gray",limits=c(min.R,max.R)) + theme(plot.title=element_text(family="Times", face="bold", size=18),axis.title=element_text(family="Times", face="bold", size=18),axis.text=element_text(family="Times", face="bold", size=18))
print(fig)
dev.off()



##########################
# Print calculation time.#
##########################
cat(sprintf("Finished. (%.1f seconds)\n",round((proc.time()-startTime)[3],digits=1)))


