## R program 'RainfallRetrieval_Links.R'.
## Program for rainfall estimation using microwave links.
## Filter outliers; compute mean path-averaged 15-min rainfall intensities.
## Described in paper:
## Aart Overeem, Hidde Leijnse, Remko Uijlenhoet, 2015. Retrieval algorithm for rainfall mapping from microwave links in a cellular communication network. Atmos. Meas. Tech. Discuss., submitted.  

## Copyright (C) 2015 Aart Overeem
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program. If not, see <http://www.gnu.org/licenses/>.


# source("RainfallRetrieval_Links.R")
# Start R and go to the directory "WorkingExample". Then run this script by pasting the line above (source(...)).
rm(list=ls(all=TRUE))
startTime = proc.time()	

# Used time step in minutes:
Timestep = 15

# Create directory for output files:
dir.create(paste("LinkData_PathRainfallDepths_",Timestep,"min",sep=""))

# Directory with input files:
direc="LinkData_Corrected"


MinutesPerDay = 1440
NrStepsDay = MinutesPerDay/Timestep


# Make list of input files: 
files_link <- list.files(path = paste(direc, sep=""), all.files=FALSE, full.names=TRUE, recursive=FALSE, pattern="linkdata")
files_link = files_link[which(file.info(files_link)$size>0)]
if (length(files_link)==0)
{
	print("No files with data! Program stops.")
	stop()
}


for (B in 1:length(files_link))
{
	print(B)
   	
	# Date:
   	date = substr(files_link[B],34,41)

	# Read link data:
   	u = read.table(files_link[B],header=TRUE)
   
   	criterium = c(NA)
   	criterium = which(u$F <= -32.5)
   	# Filter (criterion) to remove outliers: $\Sigma ( \Delta P_L - \mbox{median} ( \Delta P_L) ) <$  -130 dB km$^{-1}$ filter.    
   
   	# Apply filter to remove outliers:      
   	u$ID[criterium] =NA
   	u$a[criterium] =NA
   	u$b[criterium] =NA
   	u$Am_max[criterium] =NA
   	u$Am_min[criterium] =NA
   	u$path_length[criterium] =NA
   	u$interval_number[criterium] =NA      
   	u$x_start[criterium] =NA
   	u$y_start[criterium] =NA
   	u$x_end[criterium] =NA      
   	u$y_end[criterium] =NA  
   	u$frequency[criterium] =NA     

	ID = a = b = Am_max = Am_min = path_length = interval_number = x_start = y_start = x_end = y_end = frequency = c(NA)    
   	ID = u$ID[!is.na(u$ID)]
   	a = u$a[!is.na(u$a)] 
   	b = u$b[!is.na(u$b)]
   	Am_max = u$Am_max[!is.na(u$Am_max)]    
   	Am_min = u$Am_min[!is.na(u$Am_min)]
	path_length = u$path_length[!is.na(u$path_length)]   
   	interval_number = u$interval_number[!is.na(u$interval_number)]    
   	x_start = u$x_start[!is.na(u$x_start)]    
   	y_start = u$y_start[!is.na(u$y_start)]
   	x_end = u$x_end[!is.na(u$x_end)]   
   	y_end = u$y_end[!is.na(u$y_end)]         
   	frequency = u$frequency[!is.na(u$frequency)]  

 
	# Rainfall retrieval algorithm:
   	R_min = R_max = seq(0, 0, length.out = length(Am_max))
   
   	Aa = 2.3
	cond_max = Am_max > Aa
   	R_max[cond_max] = a[cond_max]*( (Am_max[cond_max]-Aa) /path_length[cond_max])^b[cond_max]
	cond_min = Am_min > Aa
   	R_min[cond_min] = a[cond_min]*( (Am_min[cond_min]-Aa) /path_length[cond_min])^b[cond_min]
 
   	alpha = 0.33
   	R_mean = c(NA)
   	R_mean = (1 - alpha) * R_min + alpha * R_max   
   

	# Write data to file:
	dataf =	data.frame(cbind(ID,R_mean*0.25,path_length,x_start,y_start,x_end,y_end,interval_number,frequency))
	# Construct header of output file:
   	input <- c(c('ID rainfall_depth_path path_length x_start y_start x_end y_end interval_number frequency'))
	for (s in 1:NrStepsDay)
	{      
   		dataf_s = c(NA)
   		dataf_s = dataf[interval_number==s,]
		filename_data = paste("LinkData_PathRainfallDepths_",Timestep,"min/linkdata_",date,"_",s,".dat",sep="")
		if ( nrow(dataf_s)>0 )
		{
	   		writeLines(input, filename_data)
   			write.table(dataf_s,filename_data,row.names=FALSE,col.names=FALSE,append=TRUE,quote=FALSE)
		}
	}

 
	# Print warnings per day:
	print(warnings())
	# Remove warnings:
	assign("last.warning", NULL, envir = baseenv())                    


}

# Determine number of links after applying filter and rainfall retrieval algorithm:
print(length(unique(ID)))

cat(sprintf("Finished. (%.1f seconds)\n",round((proc.time()-startTime)[3],digits=1)))


