### ******************************************************************************
### ******************************************************************************
### **************************** miniDOAS evaluation: ****************************
### ******************* average reference & calibration spectra ******************
### *************** - input data recorded with LabView, dayly files - ************
### ********************* author: Joerg Sintermann, 29.10.2014 *******************
### ******************************************************************************
### ******************************************************************************

rm(list=ls())


### input data
### ******************************************************************************
### DOAS model:
DOAS.model <- "S5" ### currently implemented: S2, S3, S4, S5

### directories:
directory <- "/miniDOAS_Revaluation_basic_version" ### main DOAS directory containing folders: "1_Reference_Spectra", "0_Measurement_Data", "2_Results"
save.dir <- paste0(directory,"/1_Reference_Spectra") ### directory of folder "1_Reference_Spectra"
eval.dir <-  "/miniDOAS_Revaluation_basic_version/miniDOAS_Revaluation" ### main directory of R evaluation folder

### averaging times
eval_timezone <- "UTC" ### compatible with strptime()
time.dark <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM") ### here, edit times so that they reflect the desired averaging periods
time.ref <-  c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM") ### ...
time.ref.dark <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")
time.N2.dark.cal <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")
time.NH3.cal <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")
time.SO2.cal <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")
time.NO.cal <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")
time.N2.cal <- c("dd.mm.YYYY HH:MM", "dd.mm.YYYY HH:MM")

### calibration conditions
NH3.cal <- c(232.974, 298, 963) ### c(ppm calibration standard, cuvette temperature [K], cuvette pressure [mbar])
SO2.cal <- c(70, 292, 988) ### ...
NO.cal <- c(60.3, 292, 988)
cuvette.length <- 0.075 ### m, length of calibration cuvette
path.length <- 14.2 * 2 ### m, total light path during calibration
NH3.amb <- 0 ### ug/m3, estimated change of ambient NH3 concentration during calibration between N2 and NH3 cuvette: will be considered in calibration conditions recorded along with averaged calibration data
NO.amb <- 0 ### ...
SO2.amb <- 0 ### ...
### ******************************************************************************




################################################################################
### setting prerequisites ######################################################
################################################################################


cat("initialise")
setwd(directory)

average.rawdat <- FALSE ### not used, but required here
rawdata.dir <- paste0(directory,"/0_Measurement_Data")

time.dark <- strptime(time.dark, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.ref <- strptime(time.ref, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.ref.dark <- strptime(time.ref.dark, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.NH3.cal <- strptime(time.NH3.cal, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.SO2.cal <- strptime(time.SO2.cal, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.NO.cal <- strptime(time.NO.cal, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.N2.cal <- strptime(time.N2.cal, "%d.%m.%Y %H:%M", tz=eval_timezone)
time.N2.dark.cal <- strptime(time.N2.dark.cal, "%d.%m.%Y %H:%M", tz=eval_timezone)

### source parameters, packages & functions
### ******************************************************************************
avg.refspecs <- TRUE
source(paste0(eval.dir,"/miniDOAS_setupeval.R"))


################################################################################
### preparation ################################################################
################################################################################


### calibration concentrations
### ******************************************************************************
cat(paste("NH3 calibration of ",NH3.cal[1]," ppm (corresponds to ",round(mDOAS.cal.to.ambient(NH3.cal[1]*1000, cuvette.length, path.length),2)," ppb in a path of ",path.length," m)", sep=""))
cat(paste("SO2 calibration of ",SO2.cal[1]," ppm (corresponds to ",round(mDOAS.cal.to.ambient(SO2.cal[1]*1000, cuvette.length, path.length),2)," ppb in a path of ",path.length," m)", sep=""))
cat(paste("NO calibration of ",NO.cal[1]," ppm (corresponds to ",round(mDOAS.cal.to.ambient(NO.cal[1]*1000, cuvette.length, path.length),2)," ppb in a path of ",path.length," m)", sep=""))
NH3.amb.ppm <- ug.to.ppb(ug=NH3.amb * path.length / cuvette.length, M=17, P.air=NH3.cal[3], T.air=NH3.cal[2]) / 1000 ### ambient amount of NH3 in light path, in ppb at calibration cuvette T & P
SO2.amb.ppm <- ug.to.ppb(ug=SO2.amb * path.length / cuvette.length, M=64, P.air=SO2.cal[3], T.air=SO2.cal[2]) / 1000
NO.amb.ppm <- ug.to.ppb(ug=NO.amb * path.length / cuvette.length, M=30, P.air=NO.cal[3], T.air=NO.cal[2]) / 1000
NH3.cal[1] <- round(NH3.cal[1] + NH3.amb.ppm, 2)
SO2.cal[1] <- round(SO2.cal[1] + SO2.amb.ppm, 2)
NO.cal[1] <- round(NO.cal[1] + NO.amb.ppm, 2)


### get data file paths (should contain files including the respective defined times)
### ******************************************************************************
dir.NH3.calibration <- paste(rawdata.dir, "/", format(time.NH3.cal, format="%Y%m%d")[1],".csv", sep="")
dir.NO.calibration <- paste(rawdata.dir, "/", format(time.NO.cal, format="%Y%m%d")[1],".csv", sep="")
dir.SO2.calibration <- paste(rawdata.dir, "/", format(time.SO2.cal, format="%Y%m%d")[1],".csv", sep="")
dir.N2.calibration <- paste(rawdata.dir, "/", format(time.N2.cal, format="%Y%m%d")[1],".csv", sep="")
dir.N2.dark.measurement <- paste(rawdata.dir, "/", format(time.N2.dark.cal, format="%Y%m%d")[1],".csv", sep="")
dir.reference.measurement <- paste(rawdata.dir, "/", format(time.ref, format="%Y%m%d")[1],".csv", sep="")
dir.ref.dark.measurement <- paste(rawdata.dir, "/", format(time.ref.dark, format="%Y%m%d")[1],".csv", sep="")
dir.dark.measurement <- paste(rawdata.dir, "/", format(time.dark, format="%Y%m%d")[1],".csv", sep="")


### create result folder
### ******************************************************************************
now1 <- as.character(Sys.time())
now <- c(unlist(strsplit(unlist(strsplit(now1,split=" ",fixed=T))[1],split="-",fixed=T)), unlist(strsplit(unlist(strsplit(now1,split=" ",fixed=T))[2],split=":",fixed=T))[1:2])
folder <- paste(c("miniDOAS-ref_cal_spec-Eval",now),collapse="")
dir.create(paste(c(save.dir,"/",folder),collapse=""),recursive=T)
setwd(paste(c(save.dir,"/",folder),collapse=""))


################################################################################
### processing #################################################################
################################################################################


### dark spectrum regular measurement
### ******************************************************************************
avg <- avg.spectra(dir.dark.measurement, time.dark, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.dark <- avg[[3]]
date.dark <- avg[[4]]
dark.spec.avg <- avg[[1]]  ### potential straylight offset is accounted for in the regular data evaluation
n.dark <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.dark <- header("dark", date.dark, NA, now, spectrometer, n.dark, integr.dark, TEC.T, board.T, integr.time)


### calibration NH3 spectra
### ******************************************************************************
avg <- avg.spectra(dir.NH3.calibration, time.NH3.cal, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.NH3.cal <- avg[[3]]
date.NH3 <- avg[[4]]
NH3.spec.avg <- avg[[1]]
n.NH3 <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.NH3 <- header("NH3", date.NH3, NH3.cal, now, spectrometer, n.NH3, integr.NH3.cal, TEC.T, board.T, integr.time)


### calibration NO spectra
### ******************************************************************************
avg <- avg.spectra(dir.NO.calibration, time.NO.cal, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.NO.cal <- avg[[3]]
date.NO <- avg[[4]]
NO.spec.avg <- avg[[1]]
n.NO <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.NO <- header("NO", date.NO, NO.cal, now, spectrometer, n.NO, integr.NO.cal, TEC.T, board.T, integr.time)


### calibration SO2 spectra
### ******************************************************************************
avg <- avg.spectra(dir.SO2.calibration, time.SO2.cal, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.SO2.cal <- avg[[3]]
date.SO2 <- avg[[4]]
SO2.spec.avg <- avg[[1]]
n.SO2 <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.SO2 <- header("SO2", date.SO2, SO2.cal, now, spectrometer, n.SO2, integr.SO2.cal, TEC.T, board.T, integr.time)


### calibration N2 spectra
### ******************************************************************************
avg <- avg.spectra(dir.N2.calibration, time.N2.cal, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.N2.cal <- avg[[3]]
date.N2 <- avg[[4]]
N2.spec.avg <- avg[[1]]
n.N2 <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.N2 <- header("N2", date.N2, NA, now, spectrometer, n.N2, integr.N2.cal, TEC.T, board.T, integr.time)


### calibration dark spectra
### ******************************************************************************
avg <- avg.spectra(dir.N2.dark.measurement, time.N2.dark.cal, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.N2.dark.cal <- avg[[3]]
date.N2.dark <- avg[[4]]
N2.dark.spec.avg <- avg[[1]]
n.N2.dark <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.N2.dark <- header("dark-N2", date.N2.dark, NA, now, spectrometer, n.N2.dark, integr.N2.dark.cal, TEC.T, board.T, integr.time)


### reference lamp spectra
### ******************************************************************************
avg <- avg.spectra(dir.reference.measurement, time.ref, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.ref <- avg[[3]]
date.ref <- avg[[4]]
ref.spec.avg <- avg[[1]]
n.ref <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.ref <- header("ref", date.ref, NA, now, spectrometer, n.ref, integr.ref, TEC.T, board.T, integr.time)


### reference lamp dark spectra
### ******************************************************************************
avg <- avg.spectra(dir.ref.dark.measurement, time.ref.dark, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3)
integr.ref.dark <- avg[[3]]
date.ref.dark <- avg[[4]]
ref.dark.spec.avg <- avg[[1]]
n.ref.dark <- avg[[2]]
TEC.T <- avg[[5]]
board.T <- avg[[6]]
integr.time <- avg[[7]]
header.ref.dark <- header("dark-ref", date.ref.dark, NA, now, spectrometer, n.ref.dark, integr.ref.dark, TEC.T, board.T, integr.time)


### save averaged spectra
### ******************************************************************************
write.table(c(header.NH3,NH3.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_NH3calSpec_",date.NH3,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(c(header.SO2,SO2.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_SO2calSpec_",date.SO2,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(c(header.NO,NO.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_NOcalSpec_",date.NO,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(c(header.ref,ref.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_refLampSpec_",date.ref,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=",", row.names=F, col.names=F, quote=F)
write.table(c(header.ref.dark,ref.dark.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_refdarkLampSpec_",date.ref,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=",", row.names=F, col.names=F, quote=F)
write.table(c(header.dark,dark.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_darkSpec_",date.dark,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(c(header.N2,N2.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_N2calSpec_",date.N2,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(c(header.N2.dark,N2.dark.spec.avg), file=paste(c(save.dir,"/",folder,"/miniDOAS_darkN2calSpec_",date.N2.dark,"-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F, quote=F)
write.table(f, file=paste(c(save.dir,"/",folder,"/Spectrometer-wavelengths-Eval",now,".txt"),collapse=""), sep=";", na="NaN", dec=".", row.names=F, col.names=F)


################################################################################
### post-processing (for plotting only) ########################################
################################################################################


### calibration DOAS curves (= high-pass filtered differential calibration spectra, for plotting only)
### ******************************************************************************
dark.offset.N2 <- mean(ref.dark.spec.avg[x3]) - mean(N2.dark.spec.avg[x3])
dark.offset.NH3 <- mean(NH3.spec.avg[x3]) - mean(N2.dark.spec.avg[x3])
dark.offset.NO <- mean(NO.spec.avg[x3]) - mean(N2.dark.spec.avg[x3])
dark.offset.SO2 <- mean(SO2.spec.avg[x3]) - mean(N2.dark.spec.avg[x3])

I.NH3 <- NH3.spec.avg - N2.dark.spec.avg
I.SO2 <- SO2.spec.avg - N2.dark.spec.avg
I.NO <- NO.spec.avg - N2.dark.spec.avg
I.N2 <- N2.spec.avg - N2.dark.spec.avg

I.NH3 <- I.NH3 / linearity.func(I.NH3, linearity.coeffs)
I.SO2 <- I.SO2 / linearity.func(I.SO2, linearity.coeffs)
I.NO <- I.NO / linearity.func(I.NO, linearity.coeffs)
I.N2 <- I.N2 / linearity.func(I.N2, linearity.coeffs)

I.NH3 <- I.NH3 - dark.offset.NH3
I.SO2 <- I.SO2 - dark.offset.SO2
I.NO <- I.NO - dark.offset.NO
I.N2 <- I.N2 - dark.offset.N2

NH3.cal.ug <- ppb.to.ug(ppb=NH3.cal[1]*1000, M=17, P.air=NH3.cal[3], T.air=NH3.cal[2])
SO2.cal.ug <- ppb.to.ug(ppb=SO2.cal[1]*1000, M=64, P.air=SO2.cal[3], T.air=SO2.cal[2])
NO.cal.ug <- ppb.to.ug(ppb=NO.cal[1]*1000, M=30, P.air=NO.cal[3], T.air=NO.cal[2])

NH3.doascurve <- doas.curve(I.NH3, I.N2, x1, filter.type, filter.strength)[[3]] / (NH3.cal.ug * cuvette.length)
NH3.diffspec <- doas.curve(I.NH3, I.N2, x1, filter.type, filter.strength)[[1]]
SO2.doascurve <- doas.curve(I.SO2, I.N2, x1, filter.type, filter.strength)[[3]] / (SO2.cal.ug * cuvette.length)
SO2.diffspec <- doas.curve(I.SO2, I.N2, x1, filter.type, filter.strength)[[1]]
NO.doascurve <- doas.curve(I.NO, I.N2, x1, filter.type, filter.strength)[[3]] / (NO.cal.ug * cuvette.length)
NO.diffspec <- doas.curve(I.NO, I.N2, x1, filter.type, filter.strength)[[1]]

#### plot calibration DOAS curves
### ******************************************************************************
plot.name <- paste(c(save.dir,"/",folder,"/miniDOAS-refspectra-Eval",now,"-"), collapse="")
pdf(file=paste(c(plot.name,"calDOAScurves.pdf"), collapse=""), width=1.25*7, height=1*7)
plot.calibration.DOAScurves(calref.cols, f[x1], f[x2], NH3.doascurve[x4], SO2.doascurve[x4], NO.doascurve[x4])

### plot callibration I/I0
### ******************************************************************************
pdf(file=paste(c(plot.name,"calDiffspecs.pdf"), collapse=""), width=1.25*7, height=1*7)
plot.calibration.diffspecs(calref.cols, f[x1], NH3.diffspec[x1], NO.diffspec[x1], SO2.diffspec[x1])

### plot reference lamp spectrum and dark spectrum + calibration spectra
### ******************************************************************************
pdf(file=paste(c(plot.name,"calRAWcurves.pdf"), collapse=""), width=1.25*7, height=1*7)
plot.calibration.spectra(calref.cols, f[x1], ref.spec.avg[x1], dark.spec.avg[x1], N2.spec.avg[x1], N2.dark.spec.avg[x1], NH3.spec.avg[x1], SO2.spec.avg[x1], NO.spec.avg[x1])

graphics.off()

