### ******************************************************************************
### ******************************************************************************
### ******************* functions for miniDOAS data evaluation *******************
### *************************** author: Joerg Sintermann *************************
### ********** this works with the Avaspec and QE65-pro from Jan.2014 on ********
### ******************************************************************************
### ******************************************************************************

functions.version <- "v3.0 (15.02.2016)"
cat("\nload functions")


### request measurement data files
### ******************************************************************************
ask.data <- function(evalperiod) {
  if (length(evalperiod) > 1) {
    folders <- strptime(list.files(path=rawdata.dir, full.names=FALSE, pattern=".csv"), "%Y%m%d", tz=eval_timezone)
    folders <- folders[!duplicated(folders)]
    folders <- format(folders[which(folders >= evalperiod[1] & folders <= evalperiod[2])], format="%Y%m%d")
    dir.meas <- paste(paste(rawdata.dir,"/",folders, sep=""), ".csv", sep="")
  } else {
    if (Sys.info()['sysname'] == "Darwin") {
      dir.meas <- file.choose() #tkchoose.files(default=directory, filters=NULL)
    } else {
      dir.meas <- file.choose() #choose.files(default=directory)
    }
  }
  return(dir.meas)
}



### open device depending on OS
### ******************************************************************************
sys.dev <- function(wi, he) {
  if (Sys.info()['sysname'] == "Darwin") {quartz(width=wi, height=he)}
  if (Sys.info()['sysname'] == "Windows") {windows(width=wi, height=he)}
}



### describe spectrometer's linearity (see Ocean Optics calibration sheets)
### ******************************************************************************
linearity.func <- function(x, cfs){
  y <- cfs[1]
  for (i in 1:(length(cfs)-1)) {
    y <- y + cfs[i+1] * x^i
  }
  return(y)
}



### ******************************************************************************
determine_full_interval <- function(x, period) {
    if (as.numeric(x, units="secs") %% as.numeric(period, units="secs") != 0) {
        y <- align.time(x, n=as.numeric(period, units="secs"))
    } else {
        y <- x
    }
    return(y)
}



### read and process calibration & reference files
### ******************************************************************************
read.references <- function() {

  ### directories of average reference & calibration files
  ### ******************************************************************************
  all.files <- list.files(path=reference.dir, pattern="\\miniDOAS_", full.names=FALSE)
  all.filedirs <- list.files(path=reference.dir, pattern="\\miniDOAS_", full.names=TRUE)
  dir.ref <- all.filedirs[grep(pattern="refLamp", x=all.files, fixed=TRUE)]                           									# reference (lamp, zero) spectrum
  dir.ref.dark <- all.filedirs[grep(pattern="refdarkLamp", x=all.files, fixed=TRUE)]                                   # reference lamp dark spectrum
  dir.dark <- all.filedirs[grep(pattern="darkSpec", x=all.files, fixed=TRUE)]                            							# measurement "dark" spectrum (BK7 or cap)
  dir.NH3.cal <- all.filedirs[grep(pattern="NH3calSpec", x=all.files, fixed=TRUE)]               											# NH3 calibration spectrum
  dir.SO2.cal <- all.filedirs[grep(pattern="SO2calSpec", x=all.files, fixed=TRUE)]                											# SO2 calibration spectrum
  dir.NO.cal <- all.filedirs[grep(pattern="NOcalSpec", x=all.files, fixed=TRUE)]                  											# NO calibration spectrum
  dir.N2.cal <- all.filedirs[grep(pattern="_N2calSpec", x=all.files, fixed=TRUE)]                 											# N2 calibration spectrum (lamp, zero, with N2 cuvette)
  dir.N2.dark.cal <- all.filedirs[grep(pattern="darkN2calSpec", x=all.files, fixed=TRUE)]       												# N2 calibration dark spectrum (with N2 cuvette)

  ### determine calibration conditions from header information (comment)
  ### ******************************************************************************
  NH3.cal <- as.numeric(unlist(strsplit(x=unlist(strsplit(x=read.table(dir.NH3.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=2)[1,], split="): ", fixed=TRUE))[2], split=",", fixed=TRUE)))
  SO2.cal <- as.numeric(unlist(strsplit(x=unlist(strsplit(x=read.table(dir.SO2.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=2)[1,], split="): ", fixed=TRUE))[2], split=",", fixed=TRUE)))
  NO.cal <- as.numeric(unlist(strsplit(x=unlist(strsplit(x=read.table(dir.NO.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=2)[1,], split="): ", fixed=TRUE))[2], split=",", fixed=TRUE)))
  cuvette.length <- as.numeric(unlist(strsplit(x=unlist(strsplit(x=read.table(dir.NH3.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=3)[1,], split="): ", fixed=TRUE))[2], split=",", fixed=TRUE)))
  reference.times <- substring(dir.ref, nchar(dir.ref)-45, nchar(dir.ref)-21)

  ### read (average) reference, calibration & (dark) noise spectra
  ### ******************************************************************************
  ### lamp reference spectrum
  dat.ref <- as.numeric(read.table(dir.ref, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### lamp reference dark spectrum
  dat.ref.dark <- as.numeric(read.table(dir.ref.dark, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### actual dark spectrum
  dat.dark <- as.numeric(read.table(dir.dark, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### NH3 calibration spectrum
  dat.NH3 <- as.numeric(read.table(dir.NH3.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### SO2 calibration spectrum
  dat.SO2 <- as.numeric(read.table(dir.SO2.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### NO calibration spectrum
  dat.NO <- as.numeric(read.table(dir.NO.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### N2 calibration reference spectrum
  dat.N2 <- as.numeric(read.table(dir.N2.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])
  ### cuvette calibration dark spectrum
  dat.N2.dark <- as.numeric(read.table(dir.N2.dark.cal, sep="\t", stringsAsFactors=FALSE, fill=TRUE, skip=10)[spec.index,])

  NH3.cal.ug <- ppb.to.ug(ppb=NH3.cal[1]*1000, M=17, P.air=NH3.cal[3], T.air=NH3.cal[2])
  SO2.cal.ug <- ppb.to.ug(ppb=SO2.cal[1]*1000, M=64, P.air=SO2.cal[3], T.air=SO2.cal[2])
  NO.cal.ug <- ppb.to.ug(ppb=NO.cal[1]*1000, M=30, P.air=NO.cal[3], T.air=NO.cal[2])

  ### dark-corrected reference spectra
  ### ******************************************************************************
  I.ref <- dat.ref - dat.ref.dark
  I.N2 <- dat.N2 - dat.N2.dark
  I.NH3 <- dat.NH3 - dat.N2.dark
  I.SO2 <- dat.SO2 - dat.N2.dark
  I.NO <- dat.NO - dat.N2.dark

  ### correct raw ccd data linearity
  ### ******************************************************************************
  I.ref <- I.ref / linearity.func(I.ref, linearity.coeffs)
  I.NH3 <- I.NH3 / linearity.func(I.NH3, linearity.coeffs)
  I.SO2 <- I.SO2 / linearity.func(I.SO2, linearity.coeffs)
  I.NO <- I.NO / linearity.func(I.NO, linearity.coeffs)
  I.N2 <- I.N2 / linearity.func(I.N2, linearity.coeffs)

  ### straylight-corrected reference spectra
  ### ******************************************************************************
  if (straylight.offset) {
      I.ref <- I.ref - mean(I.ref[x3])
      I.N2 <- I.N2 - mean(I.N2[x3])
      I.NH3 <- I.NH3 - mean(I.NH3[x3])
      I.SO2 <- I.SO2 - mean(I.SO2[x3])
      I.NO <- I.NO - mean(I.NO[x3])
  }

  ### calibration DOAS curves (= high-pass filtered differential calibration spectra)
  ### ******************************************************************************
  NH3.doascurve <- as.numeric(doas.curve(I.NH3, I.N2, x1, filter.type, filter.strength)[[3]]) / (NH3.cal.ug * cuvette.length)
  NH3.diffspec <- as.numeric(doas.curve(I.NH3, I.N2, x1, filter.type, filter.strength)[[1]])
  SO2.doascurve <- as.numeric(doas.curve(I.SO2, I.N2, x1, filter.type, filter.strength)[[3]]) / (SO2.cal.ug * cuvette.length)
  SO2.diffspec <- as.numeric(doas.curve(I.SO2, I.N2, x1, filter.type, filter.strength)[[1]])
  NO.doascurve <- as.numeric(doas.curve(I.NO, I.N2, x1, filter.type, filter.strength)[[3]]) / (NO.cal.ug * cuvette.length)
  NO.diffspec <- as.numeric(doas.curve(I.NO, I.N2, x1, filter.type, filter.strength)[[1]])

  return(
    list(
      list(NH3.cal.ug, SO2.cal.ug, NO.cal.ug, NH3.cal, SO2.cal, NO.cal, cuvette.length),
      list(I.ref, I.NH3, I.SO2, I.NO, I.N2),
      list(NH3.doascurve, SO2.doascurve, NO.doascurve),
      reference.times,
      list(NH3.diffspec, NO.diffspec, SO2.diffspec),
      list(dat.dark, dat.ref, dat.N2, dat.N2.dark, dat.NH3, dat.SO2, dat.NO)
    )
  )

}



### double moving average lowpass filter
### ******************************************************************************
doubleaverage.filter <- function(x.dat, filter.strength) {
	y <- rev(stats::filter(rev(stats::filter(x.dat, rep(1/filter.strength,filter.strength), "convolution", 2, circular=FALSE)), rep(1/filter.strength,filter.strength), "convolution", 2, circular=FALSE))
	return(y)
}



### low pass filter based on R's 'loess' function (local polynomial fitting)
### ******************************************************************************
loess.filter <- function(x.dat, filter.strength, deg=2, fam="gaussian") {
  pred <- 1:length(x.dat)
  y <- predict(loess(x.dat ~ pred, span=filter.strength, family=fam, degree=deg), data.frame(pred, se=TRUE))
  return(y)
}



### low pass filter based on R-package IDPmisc's 'rfbaseline' function (asymmetric local polynomial fitting)
### ******************************************************************************
rfbaseline.filter <- function(x.dat, filter.strength, maxit=c(2,2)) {
  pred <- 1:length(x.dat)
  y <- -rfbaseline(x=pred, y=-x.dat, span=filter.strength, maxit=maxit, Scale=function(r) mad(r))$fit
  return(y)
}



### low pass filter based on a combination of loess (gaussian) and rfbaseline function to fit a baseline through the high-pass filtered spectrum
### ******************************************************************************
rfbaseline.smooth.filter <- function(x.dat, filter.strength2=0.175, filter.strength3=0.2, maxit=c(2,2), deg=2) {
  pred <- 1:length(x.dat)
  y <- loess.filter(rfbaseline.filter(x.dat, filter.strength2, maxit=maxit), filter.strength=filter.strength3, deg=deg)
  return(y)
}


### low pass filter (for broadband absorption determination) (input: x = data to be filtered, filter.type = currently "doubleAVG" = a 2-sided moving average - or "doublehammingAVG" = a 2 sided moving hamming-average, filter.strength = strength of applied filter [data points] (uneven for "doubleAVG"))
### ******************************************************************************
lowpass.filter <- function(x.dat, filter.type, filter.strength) {
	if (filter.type == "doubleAVG") {
		y <- doubleaverage.filter(x.dat, filter.strength)
	}
	if (filter.type == "loess") {
	  y <- loess.filter(x.dat, filter.strength)
	}
	if (filter.type == "rfbaseline") {
	  y <- rfbaseline.filter(x.dat, filter.strength)
	}
	if (filter.type == "rfbaseline.smooth") {
	  y <- rfbaseline.smooth.filter(x.dat, filter.strength2=filter.strength, filter.strength3=0.15, maxit=c(2,2), deg=1)
	}
	return(y)
}



### high pass filter (for broadband absorption exclusion) (simply makes use of "lowpass.filter"; y.dat = data to be filtered, x.data = data to be filtered against - in case of moving averages x.dat should be set = y.dat -, ...)
### ******************************************************************************
highpass.filter <- function(y.dat, x.dat, filter.type, filter.strength) {
	if (filter.type == "doubleAVG") {
		y <- y.dat / doubleaverage.filter(x.dat, filter.strength)
	}
	if (filter.type == "loess") {
	  y <- y.dat / loess.filter(x.dat, filter.strength)
	}
	if (filter.type == "rfbaseline") {
	  y <- rfbaseline.filter(x.dat, filter.strength)
	}
	if (filter.type == "rfbaseline.smooth") {
      y <- x.dat / rfbaseline.smooth.filter(x.dat, filter.strength2=filter.strength, filter.strength3=0.15, maxit=c(2,2), deg=1)
	  y <- y / rfbaseline.smooth.filter(y, filter.strength2=filter.strength, filter.strength3=0.2, maxit=c(2,2), deg=2)
	}
	return(y)
}



### read (summed) rawdata, extract header, complete the averaging, for Avaspec: scale darkspectrum by darkened pixel window's offset and sd, for parallel computing...
### ******************************************************************************
read.rawdata <- function(dir.meas, x3, n.line, header.rows, pixel.number, split.date.char, split.date.format, spec.line, res.line, integr.line, board.Temp.line, dat.dark) {
  ### read data
  rawdat.meas <- fread(dir.meas, sep=";", header=FALSE, stringsAsFactors=FALSE, data.table=FALSE)
  header <- rawdat.meas[1:header.rows ,2:ncol(rawdat.meas)]
  rawdat.meas <- rawdat.meas[(1+header.rows):nrow(rawdat.meas) ,2:ncol(rawdat.meas)]
  rawdat.meas <- apply(rawdat.meas, 2, as.numeric)
  ### average summed data
  n <- as.numeric(header[n.line,])
  dat.meas <- sweep(rawdat.meas, 2, n, "/")
  ### dark count offset
  if (straylight.offset) {
      dark.offset <- apply(dat.meas[x3,], 2, function(x) mean(x) - mean(dat.dark[x3]))
  } else {
      dark.offset <- 0
  }
  ### read TEC Temperature, etc
  TEC.Temp <- as.numeric(header[TEC.line,])
  spectrometer <- header[spec.line,]
  time.res <- as.numeric(header[res.line,])
  integr.time <- as.numeric(header[integr.line,])
  board.Temp <- header[board.Temp.line,]
  ### derive date & time from file header
  splitdate <- function(x) {
  	y <- unlist(strsplit(x=x, split=split.date.char, fixed=TRUE))
  	return(y[length(y)])
  }
  date.string1 <- apply(header[starttime.line,], 2, splitdate)
  date.string2 <- apply(header[endtime.line,], 2, splitdate)
  datetime.start <- format(strptime(date.string1, split.date.format, tz=eval_timezone), format="%d.%m.%Y %H:%M:%S")
  datetime.end <- format(strptime(date.string2, split.date.format, tz=eval_timezone), format="%d.%m.%Y %H:%M:%S")
  datetime.start.leap <- substring(date.string1,nchar(date.string1)-2,nchar(date.string1))
  datetime.end.leap <- substring(date.string2,nchar(date.string2)-2,nchar(date.string2))
  return(
    list(
      list(n, dark.offset, TEC.Temp, datetime.start, datetime.end, datetime.start.leap, datetime.end.leap, spectrometer, time.res, integr.time, board.Temp),
      dat.meas
    )
  )
}



### calculate DOAS curve and intermediate steps, for parallel computing...
### ******************************************************************************
doas.curve <- function(I.meas, I.ref, x1, filter.type, filter.strength) {
  meas.broadband <- lowpass.filter(I.meas[x1], filter.type, filter.strength)
  meas.diffspec <- I.meas / I.ref
  meas.diffspec.broadband <- lowpass.filter(meas.diffspec[x1], filter.type, filter.strength)
  meas.doascurve1 <- highpass.filter(meas.diffspec[x1], meas.diffspec[x1], filter.type, filter.strength)
  meas.doascurve <- log(ifelse(meas.doascurve1 <= 0, NA, meas.doascurve1))
  meas.doascurve <- ifelse(is.na(meas.doascurve), 0, meas.doascurve)
  return(
    list(meas.diffspec, meas.diffspec.broadband, meas.doascurve, meas.broadband)
  )
}


### corrected Akaike information criterion; x = arima() output
### ******************************************************************************
AkaikeICc <- function(x) {
    ll <- logLik(x)
    no <- length(x2)#attr(x, "nobs")
    df <- attr(ll, "df")
    return((-2 * c(ll)) + (2 * df) * (1 + ((df + 1)/(no - df - 1))))
}


### fit an arima model
### ******************************************************************************
fit.arima <- function(x, NH3, SO2, NO, p, d, q) {
    y <- try(arima(x=x, order=c(p,d,q), xreg=cbind(NH3, SO2, NO), include.mean=FALSE))
    if (inherits(y,"try-error")){cat(i);y <- try(arima(x=x,  order=c(0,0,0), xreg=cbind(NH3, SO2, NO), include.mean=FALSE, method="ML"))}
    return(y)
}


### fit DOAS curves to data; either ordinary least square minimisation, or arima autoregressive model (for AR structures in residuals)
### ******************************************************************************
fit.model <- function(x, NH3, SO2, NO, type="arima", max.order) {
    if (type == "arima") {
        if (is.null(max.order)) {
            y <- list(bestmodel=fit.arima(x, NH3, SO2, NO, max.order.default$p, max.order.default$d, max.order.default$q), bestorder=paste(max.order.default,collapse="/"))
        } else {
            y <- permutate.arima(x, NH3, SO2, NO, max.order)
        }
    }
    if (type == "lm") {
        y <- list(bestmodel=lm(x ~ NH3 + SO2+ NO - 1, weights=fit.weights, model=FALSE), bestorder="")
    }
    return(y)
}


### permuate arima fits over range of order combinations and select the best model according to AICc
### ******************************************************************************
permutate.arima <- function(dat, NH3, SO2, NO, max.order) {
    arima.orders <- rbind(
        expand.grid(0:max.order$p, 0, 0, KEEP.OUT.ATTRS=FALSE),
        expand.grid(0, 0:max.order$d, 0, KEEP.OUT.ATTRS=FALSE),
        expand.grid(0, 0, 0:max.order$q, KEEP.OUT.ATTRS=FALSE)
    )
    arima.orders <- arima.orders[!duplicated(arima.orders),]
    allmodels <- apply(arima.orders, 1, function(x) fit.arima(dat, NH3, SO2, NO, p=x[1], d=x[2], q=x[3]))
    aicc <- lapply(allmodels, AkaikeICc)
    rel.likelihood <- exp((aicc[[which.min(aicc)]] - unlist(aicc)) / 2)
    return(list(bestmodel=allmodels[[which.max(rel.likelihood)]], bestorder=paste(arima.orders[which.max(rel.likelihood),], collapse="/")))
}


### retrieve goodness of fit parameter
### ******************************************************************************
fit.goodness <- function(x, type="arima") {
    if (type == "arima") {
        y <- as.numeric(x$loglik) ### change that into something better..
    }
    if (type == "lm") {
        y <- as.numeric(summary(x)$"adj.r.squared")
    }
    return(y)
}


### multiple linear fit (see Stutz & Platt, 1996, Applied Optics 30):
### fit calibration curves to measured DOAS curve, for parallel computing...
### determine best fit after shifting over given tau range and optionally consider fixed (residual) pattern beforehand
### ******************************************************************************
fit.curves <- function(meas.doascurve, meas.diffspec, x2, x4, dyn.fixed.pattern, NH3.doascurve, SO2.doascurve, NO.doascurve, fit.weights, tau.shift, tau.divisor, fit.type, max.order) {
  taus <- (-tau.shift/tau.divisor):(tau.shift/tau.divisor)
  fitcurves <- lapply(as.list(taus), function(x) fit.model(as.numeric(meas.doascurve[x2 + x*tau.divisor] - dyn.fixed.pattern), NH3.doascurve[x4],  SO2.doascurve[x4], NO.doascurve[x4], fit.type, max.order))
  fitgoodness <- as.numeric(lapply(fitcurves, function(x) fit.goodness(x$"bestmodel", fit.type)))
  ### per default: regression and residual spectrum at tau=0
  fitcurves.zero <- fitcurves[[which(taus == 0)]]
  zero.fitgoodness <- fitgoodness[which(taus == 0)]
  ### select fit with maximum goodness of fit and corresponding tau
  best.tau <- taus[which.max(fitgoodness)] * tau.divisor
  best.fitgoodness <- max(fitgoodness)
  order.best <- fitcurves[[which(taus == best.tau)]]$"bestorder"
  fitcurves.best <- fitcurves[[which(taus == best.tau)]]$"bestmodel"
  fitted.doascurve.best <- coefficients(fitcurves.best)["NH3"] * NH3.doascurve + coefficients(fitcurves.best)["SO2"] * SO2.doascurve + coefficients(fitcurves.best)["NO"] * NO.doascurve
  ### corresponding residual spectrum
  residual.best <- as.numeric(fitcurves.best$residual) #meas.doascurve[x2 + best.tau] - dyn.fixed.pattern - fitted.doascurve.best[x4]
  ### linear fit over differential spectrum
  diffspec.fit <- lm(meas.diffspec[x2 + best.tau] ~ f[x2])
  ### record meas.doascurve over all averaging intervals
  avg.meas.doascurve <- meas.doascurve[x2 + best.tau] - dyn.fixed.pattern
  ### for record fitted.doascurve.best over all averaging intervals
  coeffs <- coefficients(fitcurves.best)
  avg.fitted.NH3 <- coeffs["NH3"] * NH3.doascurve[x4]
  avg.fitted.SO2 <- coeffs["SO2"] * SO2.doascurve[x4]
  avg.fitted.NO <- coeffs["NO"] * NO.doascurve[x4]
  fitted.doascurve.best <- avg.fitted.NH3 + avg.fitted.SO2 + avg.fitted.NO
  avg.fitted.doascurve.best <- fitted.doascurve.best
  ### for memory reasons...
  fitcurves.best <- list(coefficients(fitcurves.best), vcov(fitcurves.best), residual.best)
  attr(fitcurves.best, "names")  <- c("coefficients","vcov","residuals")
  diffspec.fit <- list(diffspec.fit$coefficients, summary(diffspec.fit)$r.squared)
  attr(diffspec.fit, "names")  <- c("coefficients","r.squared")
  return(
    list(best.tau, best.fitgoodness, fitted.doascurve.best, fitcurves.best, residual.best, zero.fitgoodness, avg.fitted.doascurve.best, avg.fitted.NH3, avg.fitted.SO2, avg.fitted.NO, diffspec.fit, order.best)
  )
}


### calculate concentrations, fit parameter, etc.
### ******************************************************************************
calc.conc <- function(fitcurves.best, path.length) {
  coeffs <- unlist(lapply(fitcurves.best, coefficients))
  cov.matrix <- lapply(fitcurves.best, "[[", "vcov")
  diagonal.cov.matrix <- unlist(lapply(cov.matrix, diag))
  residual <- unlist(lapply(fitcurves.best, "[[", "residuals"))
  ### calculate concentrations (in ug/m3)
  NH3 <- coeffs[names(coeffs) == "NH3"] / path.length
  SO2 <- coeffs[names(coeffs) == "SO2"] / path.length
  NO <- coeffs[names(coeffs) == "NO"] / path.length
  ### calculate sigmas (= the standard error of individual fit coefficients) (in ug/m3) = standard error of the linear fit coefficients derived from the covariance matrix (see Stutz & Platt, 1996, Applied Optics 30)
  NH3.sig <- sqrt(diagonal.cov.matrix[names(diagonal.cov.matrix) == "NH3"]) / path.length
  SO2.sig <- sqrt(diagonal.cov.matrix[names(diagonal.cov.matrix) == "SO2"]) / path.length
  NO.sig <- sqrt(diagonal.cov.matrix[names(diagonal.cov.matrix) == "NO"]) / path.length
  ### partial correlation coefficients from multiple linear regression (see ???)
  t.values <- coeffs / sqrt(diagonal.cov.matrix)
  partcorr <- sqrt((t.values^2) / ((t.values^2) + length(residual)-1))
  partcorr <- cbind(partcorr[names(partcorr) == "NH3"], partcorr[names(partcorr) == "SO2"], partcorr[names(partcorr) == "NO"])
  return(list(NH3, SO2, NO, NH3.sig, SO2.sig, NO.sig, partcorr))
}


### ******************************************************************************
number.density <- function(pressure, volume, temperature) { ### pressure = Pa, volume = m3, temperature = K; output = number of molecules
            R <- 8.3144621
            Avogadro <- 6.022 * 10^23
            n <- pressure * volume / R / temperature * Avogadro
            return(n)
}



### averaging ref/cal. files
### ******************************************************************************
avg.spectra <- function(dir.meas, avg.time, pixel.number, n.line, integr.line, TEC.line, board.Temp.line, split.date.char, split.date.format, header.rows, x3) {
  dat.meas <- read.rawdata(dir.meas, x3, n.line,header.rows, pixel.number, split.date.char, split.date.format, spec.line, res.line, integr.line,  board.Temp.line, array(data=0, dim=pixel.number))
  n <- dat.meas[[1]][[1]]; TEC.Temp <- dat.meas[[1]][[3]]; datetime.start <- dat.meas[[1]][[4]]; datetime.end <- dat.meas[[1]][[5]]; datetime.start.frac <- dat.meas[[1]][[6]]; datetime.end.frac <- dat.meas[[1]][[7]]; spectrometer <- dat.meas[[1]][[8]]; time.res <- dat.meas[[1]][[9]]; integr.time <- dat.meas[[1]][[10]]; board.Temp <- dat.meas[[1]][[11]]; dat.meas <- dat.meas[[2]]
  datetime.start <- strptime(datetime.start, "%d.%m.%Y %H:%M:%S", tz=eval_timezone)
  datetime.end <- strptime(datetime.end, "%d.%m.%Y %H:%M:%S", tz=eval_timezone)
  date <- paste(format(datetime.start[which(datetime.start >= avg.time[1])[1]], format="%Y%m%d%H%M"), "-", format(datetime.end[tail(which(datetime.end <= avg.time[2]), 1)], format="%Y%m%d%H%M"), sep="")
  ### average spectra of given calibration times
  index <- which(datetime.start >= avg.time[1] & datetime.end <= avg.time[2])
  if (length(index) == 1) {spec.avg <- dat.meas[,index]} else {spec.avg <- rowMeans(dat.meas[,index], na.rm=TRUE)}
  n.specs <- length(index)
  n <- round(mean(n[index], na.rm=TRUE), 0)
  integr.time <- round(mean(integr.time[index], na.rm=TRUE), 0)
  TEC.Temp <- round(mean(ifelse(TEC.Temp > 99 | TEC.Temp == -0.1 | is.na(TEC.Temp), NA, TEC.Temp[index]), na.rm=TRUE), 2)
  board.Temp <- round(mean(as.numeric(unlist(lapply(as.list(board.Temp[index]), function(x) strsplit(x, split=",", fixed=TRUE))))[seq(1,length(index)*3,3)]),2)
  return(list(spec.avg, n.specs, n, date, TEC.Temp, board.Temp, integr.time))
}



### header for averaged ref/cal. files
### ******************************************************************************
header <- function(what, date, cal, now, spectrometer, n.specs, n, TEC.T, board.T, integr.time) {
  if (what == "NH3" | what == "NO" | what == "SO2" | what == "N2") {
    conditions <- paste(c("conditions: calibration measurement of ",what," (concentration in ppm, temperature in K, pressure in mbar): ",cal[1],",",cal[2],",",cal[3]), collapse="")
  }
  if (what == "dark") {
    conditions <- "conditions: dark measurement"
  }
  if (what == "dark-N2") {
    conditions <- "conditions: dark measurement N2"
  }
  if (what == "dark-ref") {
    conditions <- "conditions: dark measurement reference spectrum"
  }
  if (what == "ref") {
    conditions <- "conditions: reference measurement"
  }
  head <- c(
            paste(c("reference file calculation performed on: ",now),collapse=""),
            paste(c("spectrometer: ",spectrometer),collapse=""),
            conditions,
            paste(c("cuvette length (m): ",cuvette.length),collapse=""),
            paste(c("accumulation: averaged ",n.specs," avg spectra with ",n," spectra each"),collapse=""),
            paste(c("averaging period: ",date),collapse=""),
            paste(c("avg. spectrometer integration time (ms): ",integr.time),collapse=""),
            paste(c("avg. TEC temperature (deg C): ",TEC.T),collapse=""),
            paste(c("avg. DOAS temperature (deg C): ",board.T),collapse=""),
            "---"
          )

  return(head)
}


### miniDOAS: relate calibration concentration to equivalent ambient concentration for given path length (input: cal.conc = calibration concentration inside cuvette, cuvette.length = length of cuvette [m], path.length = total length of light path for ambient measurements [m])
### ******************************************************************************
mDOAS.cal.to.ambient <- function(cal.conc, cuvette.length, path.length) {
	ambient.conc <- cal.conc / (path.length / cuvette.length)
	ambient.conc
}



### convert trace gas concentration in ppb into concentration in ug m-3 (ppb=trace gas concentration in ppb; molmass=molecular mass of trace gas species in g mol-1; P.air=air pressure in mbar; T.air=air temperature in K)
### ******************************************************************************
ppb.to.ug <- function(ppb, M, P.air, T.air) {
	ug <- ((ppb / 1000000000) * M * (((P.air * 100) * 1) / (8.314472 * T.air))) * 1000000
	ug
}



### convert trace gas concentration in ug/m3 into mixing ratio ppb (ug=trace gas concentration in ug/m3; molmass=molecular mass of trace gas species in g mol-1; P.air=air pressure in mbar; T.air=air temperature in K)
### ******************************************************************************
ug.to.ppb <- function(ug, M, P.air, T.air) {
  ppb <- ug * 1000000000 / ((1000000 * (((P.air * 100) * 1) / (8.314472 * T.air))) * M )
  ppb
}



### function to draw (and annotate) (deca-)logarithmic ticks on R standard plots (plot() has to be plotted with x/yaxt="n" beforehand) (input: axis.loc = axis location: 1,2,3 or 4 for c(bottom, left, top, right), axis.values = vector of the data plotted on respective axis, tick.dim = c(major tick length, minor tick length) - eg. c(-0.5,-0.25), cex.labels = cex for axis labels)
### ******************************************************************************
log10.ticks <- function(axis.loc, axis.values, tick.dim, cex.labels) {
	axis.values <- ifelse(axis.values <= 0, NA, axis.values)
	val.range <- floor(log10(range(axis.values, na.rm=T, finite=T)))
	pow <- seq(val.range[1],val.range[2]+1)
	ticksat <- as.vector(sapply(pow, function(p) (1:10)*10^p))
	axis(axis.loc,pow,labels=10^pow,tcl=tick.dim[1],cex.axis=cex.labels)
	axis(axis.loc,log10(ticksat),labels=NA,tcl=tick.dim[2],lwd=0,lwd.ticks=1)
}



### plot raw reference (with and without cuvette), dark, and calibration spectra
### ******************************************************************************
plot.calibration.spectra <- function(calref.cols, wavelengths, lamp.reference, dark, lamp.N2, dark.N2, NH3, SO2, NO) {
	cex.annotations <- 1.25
	y.limes <- log10(range(c(lamp.reference,dark,dark.N2,lamp.N2,NO,SO2,NH3), na.rm=T))
	plot(wavelengths, log10(lamp.reference), type="l", lty=2, lwd=1, yaxt="n", xlab="wavelength [nm]", ylab="averaged signal [counts]", main="miniDOAS reference, calibration, & dark spectra", cex.axis=cex.annotations, cex.lab=cex.annotations, ylim=y.limes)
	lines(wavelengths, log10(dark), lty=1, lwd=1, col="gray30")
	lines(wavelengths, log10(dark.N2), lty=2, lwd=1, col="gray50")
	lines(wavelengths, log10(lamp.N2), lty=1, lwd=1, col=calref.cols[4])
	lines(wavelengths, log10(NO), lty=1, lwd=1, col=calref.cols[3])
	lines(wavelengths, log10(SO2), lty=1, lwd=1, col=calref.cols[2])
	lines(wavelengths, log10(NH3), lty=1, lwd=1.5, col=calref.cols[1])
	log10.ticks(2, c(lamp.reference,dark,dark.N2,lamp.N2,SO2,NO,NH3), c(-0.5,-0.25), cex.annotations)
	legend("topleft", legend=c(expression(italic("I0'")),expression(italic("dark")),expression(italic(paste(N[2]," dark",sep=""))),expression(italic(N[2])),expression(italic("NO")),expression(italic(paste("SO"[2],sep=""))),expression(italic(paste("NH"[3],sep="")))), bty="n", col=c("black","gray50","gray80",rev(calref.cols)), lty=c(2,1,2,1,1,1,1), cex=0.75, ncol=2)
}



### plot calibration DOAS curves
### ******************************************************************************
plot.calibration.DOAScurves <- function(calref.cols, wavelengths1, wavelengths2, NH3.dc, SO2.dc, NO.dc) {
	cex.annotations <- 1.25
	par(mar=c(5,4,2,2)+0.6)
	plot(wavelengths1, wavelengths1, type="n", ylim=range(c(NH3.dc,SO2.dc,NO.dc)), xlab="wavelength [nm]", ylab=expression(paste("DOAS absorption cross-section [",m^2,"/",mu,"g]",sep="")), main="calibration DOAS curves", cex.axis=cex.annotations, cex.lab=cex.annotations)
	abline(h=0,lty=2,col="gray60")
	lines(wavelengths2, NO.dc, lty=1, lwd=1.5, col=calref.cols[3])
	lines(wavelengths2, SO2.dc, lty=1, lwd=1.5, col=calref.cols[2])
	lines(wavelengths2, NH3.dc, lty=1, lwd=1.5, col=calref.cols[1])
	legend("bottomright", legend=c(expression(italic(paste("NH"[3],sep=""))),expression(italic(paste("SO"[2],sep=""))),expression(italic("NO"))), lty=c(1,1,1), cex=0.75, col=calref.cols[1:3], bty="n")
}



### plot calibration diffspec
### ******************************************************************************
plot.calibration.diffspecs <- function(calref.cols, wavelengths1, NH3.diffspec, NO.diffspec, SO2.diffspec) {
	cex.annotations <- 1.25
	par(mar=c(5,4,2,2)+0.6)
	plot(wavelengths1, wavelengths1, type="n", ylim=c(0,max(c(NH3.diffspec,NO.diffspec,SO2.diffspec))), xlab="wavelength [nm]", ylab="I/I0'", main="calibration diffspec", cex.axis=cex.annotations, cex.lab=cex.annotations)
	abline(h=1,lty=2,col="gray60")
	lines(wavelengths1, SO2.diffspec, lty=1, lwd=1.5, col=calref.cols[2])
	lines(wavelengths1, NO.diffspec, lty=1, lwd=1.5, col=calref.cols[3])
	lines(wavelengths1, NH3.diffspec, lty=1, lwd=1.5, col=calref.cols[1])
	legend("bottomright", legend=c(expression(italic(paste("NH"[3],sep=""))),expression(italic(paste("SO"[2],sep=""))),expression(italic("NO"))), lty=c(1,1,1), cex=0.75, col=calref.cols[1:3], bty="n")
}


### create log file
### ******************************************************************************
log.file <- function() {
  spectrometer <- spectrometer[!duplicated(spectrometer)][1]
  time.res <- round(mean(time.res, na.rm=TRUE), 2)
  logfile <- c(
    paste(c("miniDOAS evaluation program version: ",program.version),collapse=""),
    "*****************************************************************************",
    paste(c("calculation performed on: ",now1),collapse=""),
    ifelse(length(dir.meas) == 1,paste(c("raw data file: ",dir.meas),collapse=""),paste(c("raw data files: ",dir.meas[1]," to ",dir.meas[length(dir.meas)]," (",length(dir.meas)," files)"),collapse="")),
    paste(c("results saved at: ",save.dir,"/",folder),collapse=""),
    "",
    paste(c("reference files directory: ",reference.dir),collapse=""),
    paste(c("Iref spectrum recorded on: ",reference.times),collapse=""),
    paste(c("NH3 calibration conditions: NH3 conc = ",NH3.cal[1],"ppm @ ",NH3.cal[2]," K and ",NH3.cal[3]," mbar"),collapse=""),
    paste(c("SO2 calibration conditions: SO2 conc = ",SO2.cal[1],"ppm @ ",SO2.cal[2]," K and ",SO2.cal[3]," mbar"),collapse=""),
    paste(c("NO calibration conditions: NO conc = ",NO.cal[1],"ppm @ ",NO.cal[2]," K and ",NO.cal[3]," mbar"),collapse=""),
    paste(c("in a cuvette of ",cuvette.length," m length"),collapse=""),
    "",
    paste(c("total light path = ",path.length," m"),collapse=""),
    paste(c("spectrometer: ",spectrometer),collapse=""),
    paste(c(round(mean(nu),0)," spectra were accumulated on average over ",time.res," seconds for one data file"),collapse=""),
    "",
    paste(c("files evaluated from ",format(results[1,1],format="%d.%m.%Y %H:%M:%S")," to ",format(results[nrow(results),2],format="%d.%m.%Y %H:%M:%S")),collapse=""),
    paste(c("spectra were evaluated over ",wl.window[1]," to ",wl.window[2]," nm"),collapse=""),
    paste(c("... using a ",ifelse(filter.type=="doubleAVG","double moving average",ifelse(filter.type=="recursive","recursive",ifelse(filter.type=="doublehammingAVG","double hamming moving average",ifelse(filter.type=="loess","local fitting ('loess')","loess-smoothed rfbaseline ('rfbaseline.smooth')"))))," filter with width of ",filter.strength),collapse=""),
    paste(c("spectra were ",ifelse(corr.shift,"","not "),"allowed to be shifted",ifelse(corr.shift,paste(c(" at max. +/- ",tau.shift," pixel with a resolution of ",tau.divisor," pixel"),collapse=""),"")),collapse=""),
    paste(c("a pre-evaluated fixed residual pattern calculated by runmed over ",fixed.pattern.length," datapoints was ",ifelse(corr.fixed.pattern,"subtracted","not substracted")),collapse=""),
    paste(c("fitting was performed with ",fit.type," setting"),collapse=""),
    paste(c("(potential) stray-light mapping from nominal ",dark.wl[1]," to ",dark.wl[2]," nm"),collapse="")
  )
  return(logfile)
}


### plot (and save) average DOAS curves and average residual spectrum over defined average periods (assuming 1 minute averaged raw spectra)
### ******************************************************************************
plot.offline <- function() {

	if (!is.na(avg.period) & avg.period <= nrow(results)) {

			cat(paste("\r..saving plots... ",format(x=round(i/floor(nrow(results) / avg.period)*100,1),nsmall=1),"%",sep=""))
			avg.index <- ((i-1)*avg.period+1):(i*avg.period)
			if (corr.fixed.pattern) {if (length(avg.index) > 1) {avg.fixed.pattern <- colMeans(fp.avg[avg.index,])} else {avg.fixed.pattern <- fp.avg[avg.index,]}} else {avg.fixed.pattern <- rep(0, length(x2))}
			best.tau <- round(mean(results[avg.index,6]),0)
			if (length(avg.index) > 1) {m.meas <- rowMeans(dat.meas[x1, avg.index])} else {m.meas <- dat.meas[x1, avg.index]}
			if (length(avg.index) > 1) {m.diffspec <- rowMeans(meas.diffspec[x1,avg.index])} else {m.diffspec <- meas.diffspec[x1,avg.index]}
			if (length(avg.index) > 1) {m.diffspec.broadband <- rowMeans(meas.diffspec.broadband[x1,avg.index])} else {m.diffspec.broadband <- meas.diffspec.broadband[x1,avg.index]}
			if (length(avg.index) > 1) {m.fitted.doascurve.best <- rowMeans(avg.fitted.doascurve.best[,avg.index])} else {m.fitted.doascurve.best <- avg.fitted.doascurve.best[,avg.index]}
			if (length(avg.index) > 1) {m.meas.doascurve <- rowMeans(meas.doascurve[x2,avg.index])} else {m.meas.doascurve <- meas.doascurve[x2,avg.index]}
			if (length(avg.index) > 1) {m.meas.doascurve.best <- rowMeans(meas.doascurve[x2+best.tau,avg.index])} else {m.meas.doascurve.best <- meas.doascurve[x2+best.tau,avg.index]}
			if (length(avg.index) > 1) {m.residual <- rowMeans(residual[,avg.index])} else {m.residual <- residual[,avg.index]}
			if (length(avg.index) > 1) {m.residual.best <- rowMeans(residual.best[,avg.index])} else {m.residual.best <- residual.best[,avg.index]}

			par(mfcol=c(3,1),mar=c(4,4,3,1)+0.5,new=FALSE)
			cex.annotations <- 1.25

			### differential spectrum and filtered differential broadband spectrum
			plot(f[x1], x1, type="n", ylim=range(m.diffspec, na.rm=TRUE), xlab="", ylab="I/I0' [relative units]", main="differential spectrum", cex.axis=cex.annotations, cex.lab=cex.annotations)
			lines(f[x1], m.diffspec, lty=1, lwd=1.5, col="black")
			lines(f[x1], m.diffspec.broadband, lty=1, lwd=1.5, col="green")
			legend("bottomright", legend=c("differential","differential broadband"), bty="n", col=c("black","green"), lty=c(1,2), cex=0.75)

			### measued & fitted DOAS curve
			plot(f[x1], x1, type="n", ylim=range(c(m.fitted.doascurve.best,m.meas.doascurve-avg.fixed.pattern,m.meas.doascurve.best-avg.fixed.pattern,m.meas.doascurve), na.rm=T), xlab="", ylab="[relative units]", main="DOAS curves", cex.axis=cex.annotations, cex.lab=cex.annotations)
			abline(h=0,lty=2,col="gray60")
			if (corr.fixed.pattern) {lines(f[x2], m.meas.doascurve, lty=1, lwd=1.5, col="gold")}
			lines(f[x2], (m.meas.doascurve - avg.fixed.pattern), lty=1, lwd=1, col="gray80")
			lines(f[x2 + best.tau], (m.meas.doascurve.best - avg.fixed.pattern), lty=1, lwd=1.5, col="black")
			lines(f[x2 + best.tau], m.fitted.doascurve.best, lty=1, lwd=1.5, col="blue")
			lgnd <- if(corr.fixed.pattern) {c("measured","measured-FP+shift","measured-FP","fitted")} else{c("measured+shift","measured","fitted")}
			lgnd.lty <- if(corr.fixed.pattern) {c(1,1,1,1)} else{c(1,1,1)}
			lgnd.col <- if(corr.fixed.pattern) {c("gold","black","gray80","blue")} else{c("black","gray80","blue")}
			legend("bottomright", legend=lgnd, bty="n", col=lgnd.col, lty=lgnd.lty, cex=0.75)

			### residual spectrum
			plot(f[x1], x1, type="n", ylim=range(c(avg.fixed.pattern,m.residual,m.residual.best), na.rm=T), xlab="wavelength [nm]", ylab="[relative units]", main="residual spectrum", cex.axis=cex.annotations, cex.lab=cex.annotations)
			abline(h=0,lty=2,col="gray60")
			if (corr.fixed.pattern) {lines(f[x2], avg.fixed.pattern, lty=1, lwd=1.5, col="gold")}
			#lines(f[x2 + best.tau], m.fitted.doascurve.best, lty=2, lwd=1.5, col="blue")
			lines(f[x2], m.residual, lty=2, lwd=1, col="gray80")
			lines(f[x2+best.tau], m.residual.best, lty=1, lwd=1.5, col="black")

			### legend & title
			lgnd <- if(corr.fixed.pattern) {c("fixed pattern","fitted DOAS curve","residuals+shift","residuals")} else{c("fitted DOAS curve","residuals+shift","residuals")}
			lgnd.lty <- if(corr.fixed.pattern) {c(2,2,1,2)} else{c(2,1,2)}
			lgnd.col <- if(corr.fixed.pattern) {c("gold","blue","black","gray80")} else{c("blue","black","gray80")}
			legend("bottomright", legend=lgnd, bty="n", col=lgnd.col, lty=lgnd.lty, cex=0.75)
			msg <- paste(results[avg.index[1],1]," to ",results[avg.index[length(avg.index)],2],", n=",sum(results[avg.index,5]),", NH3=",round(mean(results[avg.index,9]),1)," +- ",round(mean(results[avg.index,10]),2)," ug/m3, SO2=",round(mean(results[avg.index,12]),1)," ug/m3, NO=",round(mean(results[avg.index,15]),1)," ug/m3, tau=",best.tau," px, R2=",round(mean(results[avg.index,7]),2), sep="")
			mtext(as.expression(substitute(italic(msg), list(msg=msg))), line=-1.25, outer=T, cex=0.5)

	}
}



