## R program 'Interpolation_Rainfall_Intensities.R'.
## Program for rainfall estimation using microwave links.
## Interpolation of link-based path-averaged rainfall intensities.
## Described in paper:
## Aart Overeem, Hidde Leijnse, Remko Uijlenhoet, 2015. Retrieval algorithm for rainfall mapping from microwave links in a cellular communication network. Atmos. Meas. Tech. Discuss., submitted.  

## Copyright (C) 2015 Aart Overeem
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program. If not, see <http://www.gnu.org/licenses/>.


# Interpolation of observed microwave link daily rainfall depths. "Van de Beek variograms" have been used. These variograms have been
# modelled as function of the day number and duration in the paper by Van de Beek et al. They use durations of 1 - 24 h. In this script the
# relationships have been extrapolated to 15-min data. For all input data the middle of the link is used as location of the path-averaged
# measurement. Input data are link-based 15-minute path-averaged rainfall accumulations. These rainfall accumulations are assumed to be
# representative for the point at the center of the link.

# source("Interpolation_Rainfall_Intensities.R")
# Start R and go to the directory "WorkingExample". Then run this script by pasting the line above (source(...)).
rm(list=ls(all=TRUE))
startTime = proc.time()             # Get starting time
   
# Used time step in minutes:
Timestep = 15

MinutesInHour = 60
ConversionRainfallDepth_To_RainfallIntensity = MinutesInHour/Timestep

# Compute duration in hours
D = Timestep/MinutesInHour
f = 1/365

# Create directory for output files:
dir.create(paste("RainfallMaps_",Timestep,"min_links",sep=""))


####################
# Load R libraries.#
####################
#pathlib = "path name"
library(sp)	#,lib.loc=pathlib)
library(gstat)


###############################
# Load grid for interpolation.#
###############################

# Read interpolation grid (should be supplied in WGS84 in degrees):
# Load R library
library(rgdal)
path.grid = "InterpolationGrid.dat"
rain.grid = read.table(path.grid,header=TRUE,sep=",")
# X_middle = center of coordinate system (longitude); Y_middle = center of coordinate system (latitude); 
# We chose 52.155223°N 5.387242°E as the middle of the Netherlands ('The Tower of Our Lady' is a church tower in Amersfoort and used to be the middle point of the Dutch grid reference system).
X_middle = 5.387242
Y_middle = 52.155223
# Convert WGS84 coordinates (degrees) to an Azimuthal Equidistant Cartesian coordinate system (easting and northing of grid; m).
# of start of link, easting and northing of end of link, respectively; km). 
d <- data.frame(lon=rain.grid$X,lat=rain.grid$Y)
coordinates(d) <- c("lon", "lat")
proj4string(d) <- CRS("+init=epsg:4326") # WGS 84
CRS.new <- CRS(paste("+proj=aeqd +a=6378.137 +b=6356.752 +R_A +lat_0=",Y_middle," +lon_0=",X_middle," +x_0=0 +y_0=0",sep=""))
UTM <- spTransform(d, CRS.new)
#UTM$lon  Easting (in km)
#UTM$lat  Northing (in km)
rain.grid = data.frame(cbind(UTM$lon,UTM$lat))
coordinates(rain.grid) = as.data.frame(rain.grid[,])


# Make list of input files: 
files_link <- list.files(path = paste("LinkData_PathRainfallDepths_",Timestep,"min",sep=""), all.files=FALSE,full.names=TRUE, recursive=FALSE, pattern="linkdata")
files_link = files_link[which(file.info(files_link)$size>0)]
if (length(files_link)==0)
{
	print("No files with data! Program stops.")
	stop()
}

NrUniqueLinks = NrUniqueLinkPaths = c(NA)
q = 0
for (filenr in 1:length(files_link))
{

	date = substr(files_link[filenr],44,51) 

	##########################################################
	# Reading measured 15-min microwave link rainfall depths.#
	##########################################################
   
	# Read path-averaged link rainfall depths:
	u = read.table(files_link[filenr], header=TRUE)   

	# Determine time interval number:
	int_number = unique(u$interval_number)
   
	# Determine coordinates of middle of link:
	X_middle = Y_middle = c(NA)
	X_middle = (u$x_start+u$x_end)/2
	Y_middle = (u$y_start+u$y_end)/2  

	# Determine unique middle of links:
	coord_dataf = data.frame(cbind(X_middle,Y_middle))
	coord_uniq = unique(coord_dataf)
	rainlink = array(NA,c(length(coord_uniq[,1]),3))

	# Compute average number of unique link paths:
	NrUniqueLinkPaths[filenr]=(length(coord_uniq[,1]))
	
	# Compute average number of unique links (full-duplex links count twice)
	NrUniqueLinks[filenr]=length(u$ID)

	# Compute mean path-averaged rainfall intensities for each unique middle of link. 
	# I.e., rainfall intensities from full-duplex links are averaged.
	rainlink[,1] = coord_uniq[,1]
	rainlink[,2] = coord_uniq[,2]
	number_linkpaths = length(coord_uniq[,1])
	for (i in 1:number_linkpaths)
	{
		q = q + 1
		rainlink[i,3] = mean(u$rainfall_depth_path[rainlink[i,1]==X_middle&rainlink[i,2]==Y_middle]) * ConversionRainfallDepth_To_RainfallIntensity
	}
  
	rainlink = as.data.frame(rainlink)
	colnames(rainlink) = c("X","Y","RAIN")
	centres = cbind(rainlink$X,rainlink$Y)
	coordinates(rainlink) = centres   



	#########################################################
	# Calculating coefficients of spherical variogram model.#
	#########################################################

	# Calculate sill and range for this particular day:
	t = strptime(date, "%Y%m%d")$yday+1  # Determines day of year (Julian day number)
	   
	range = (15.51 * D^0.09 + 2.06 * D^-0.12 * cos(2*pi*f * (t - 7.37 * D^0.22) ) )^4  
	sill = (0.84 * D^-0.25 + 0.20 * D^-0.37 * cos(2*pi*f * (t - 162 * D^-0.03) ) )^4

	C0 = 0.1 * sill # nugget
	C = sill - C0 # sill - nugget
	r = range/1000 # range (in kilometers)
   


	############################
	# Interpolate with kriging.#
	############################

	KRIGINS = krige(rainlink$RAIN~1,rainlink,rain.grid,model=vgm(sill,"Sph",r,C0),nmax=50)



	###################################################
	# Write the interpolated rainfall field to a file.#
	###################################################

	# Function for printing:
	file.name = paste("RainfallMaps_",Timestep,"min_links/kriging_",date,"_",Timestep,"min_",int_number,".txt",sep="")
	# Construct header of output file:
   	input <- c(c('rain_intensity'))
	writeLines(input, file.name)
	# Negative values are set to 0:
	KRIGINS$var1.pred[KRIGINS$var1.pred<0]=0
	write.table(KRIGINS$var1.pred,file=file.name,quote=FALSE,row.names=FALSE,col.names=FALSE,append=TRUE)


}   

# Print average number of unique link paths over those time steps which exist (have data):
print(mean(NrUniqueLinkPaths))
	
# Print average number of unique links over those time steps which exist (have data). Note that full-duplex links count twice.
print(mean(NrUniqueLinks))


##########################
# Print calculation time.#
##########################
cat(sprintf("Finished. (%.1f seconds)\n",round((proc.time()-startTime)[3],digits=1)))
   


