#pragma TextEncoding = "UTF-8"
#pragma rtGlobals=3		// Use modern global access method and strict wave access.

///////////////////////////////////////////////////////////////////////////////////////////////
//    Development of a Universal Correction Algorithm for Filter-Based Absorption Photometers//
//                  Igor Pro code for the algorithms (A,B,C)                                 //
///////////////////////////////////////////////////////////////////////////////////////////////
// 		               Code by: Hanyang Li, li.4860@osu.edu                                 //
//////////////////////////////////////////////////////////////////////////////////////////////



//load the data
LoadWave /A/Q/O/J/D/W/K=0 "C:Users:Hanyang:Desktop:FIREX_CLAP.csv"


//=====================================list of variables used in the code===================//
//reference absorption coefficient (e.g., from a Photoacoustic Extinctiometer): babs_red_ref, babs_green_ref, babs_blue_ref
//scattering coefficient (e.g., from a Nephelometer): bscat_red, bscat_green, bscat_blue

//attenuation coefficient measured by the Filter-Based Absorption Photometer: batn_red, batn_green, batn_blue
//transmission measured by the Filter-Based Absorption Photometer: Tr_red, Tr_green, Tr_blue
//========================================================================================//




function Algo_Li_et_al_2019()


if (exists("babs_red_ref")==1 && exists("babs_green_ref")==1 && exists("babs_blue_ref")==1)  // reference absorption coefficient exists

	if (exists("bscat_red")==1 && exists("bscat_green")==1 && exists("bscat_blue")==1)  // scattering coefficient exists 
	
		print "Implement Algorithm A"
		Algo_A()
		
	else

		// scattering coefficient does not exist
		print "The scenario is not considered in the manuscript. We recommend using the photoacoustic Babs."
		
	endif
	
else // reference absorption coefficient does not exist

	if (exists("bscat_red")==1 && exists("bscat_green")==1 && exists("bscat_blue")==1)  // scattering coefficient exists 
	
		print "Implement Algorithm B"
		Algo_B()
		
 	else 
 	
 		// scattering coefficient does not exist 	
		print "Implement Algorithm C"
		Algo_C()
		
	endif	
endif

end



function Algo_A()

wave babs_red_ref,babs_green_ref,babs_blue_ref,bscat_red,bscat_green,bscat_blue,batn_red,batn_green,batn_blue,Tr_red,Tr_green,Tr_blue

variable npnts=numpnts(batn_red)


//initialize AAE, using batn
AAE_power_fit(batn_red,batn_green,batn_blue)
wave AAE_three_lambda
make /N=(npnts)/O AAE_temp=AAE_three_lambda  //"_temp" means that the term is a temporary variable, which is updated in the following loop 

//initialize SSA, using batn and bscat_ref
make /N=(npnts)/O SSA_red_temp=bscat_red/(bscat_red+batn_red)
make /N=(npnts)/O SSA_green_temp=bscat_green/(bscat_green+batn_green)
make /N=(npnts)/O SSA_blue_temp=bscat_blue/(bscat_blue+batn_blue)


//initialize set of coefficients
make/N=8/O W_coef_temp_red = {0,0,0,0,0,0,0,0}
make/N=8/O W_coef_temp_green = {0,0,0,0,0,0,0,0}
make/N=8/O W_coef_temp_blue = {0,0,0,0,0,0,0,0}

//compute "g" term in Eq. (9), where g(Tr,SSA,AAE)=babs_ref/batn
make /N=(npnts)/O g_term_red_temp=babs_red_ref/batn_red
make /N=(npnts)/O g_term_green_temp=babs_green_ref/batn_green
make /N=(npnts)/O g_term_blue_temp=babs_blue_ref/batn_blue

//compute ln_Tr
make /N=(npnts)/O ln_Tr_red=ln(Tr_red)
make /N=(npnts)/O ln_Tr_green=ln(Tr_green)
make /N=(npnts)/O ln_Tr_blue=ln(Tr_blue)


variable loop_number=0
print "////////////iterative process////////////"  //show major results in the command window


do
	loop_number+=1
	print "Loop No.", loop_number

	//temporarily save the outputs from the previous iteration
	duplicate /O AAE_temp,AAE_previous_cycle  

	//implement "Algorithm A" on each wavelength
	make/D/N=8/O W_coef=W_coef_temp_red
	FuncFit/Q Algorithm_function W_coef g_term_red_temp /X={ln_Tr_red,SSA_red_temp,AAE_temp}  /w=batn_red  

		W_coef_temp_red=W_coef
		
	make/D/N=8/O W_coef =W_coef_temp_green
	FuncFit/Q Algorithm_function W_coef g_term_green_temp /X={ln_Tr_green,SSA_green_temp,AAE_temp}  /w=batn_green 
		W_coef_temp_green=W_coef

	make/D/N=8/O W_coef =W_coef_temp_blue
	FuncFit/Q Algorithm_function W_coef g_term_blue_temp /X={ln_Tr_blue,SSA_blue_temp,AAE_temp}  /w=batn_blue 
 		W_coef_temp_blue=W_coef

		if  (loop_number==1) //output coefficients for future uses
			Concatenate /O/NP=2 {W_coef_temp_red,W_coef_temp_green,W_coef_temp_blue}, W_coef_initial_guess_all_wvl		
		endif
		
	//Calculate Babs for each wavelength using Eq. (8) in the manuscript
	make /N=(npnts)/O b_abs_red_temp=batn_red*(W_coef_temp_red[0]+W_coef_temp_red[1]*ln_Tr_red+W_coef_temp_red[2]*SSA_red_temp+W_coef_temp_red[3]*AAE_temp+W_coef_temp_red[4]*ln_Tr_red*SSA_red_temp+W_coef_temp_red[5]*SSA_red_temp*AAE_temp+W_coef_temp_red[6]*ln_Tr_red*AAE_temp+W_coef_temp_red[7]*ln_Tr_red*AAE_temp*SSA_red_temp)

	make /N=(npnts)/O b_abs_green_temp=batn_green*(W_coef_temp_green[0]+W_coef_temp_green[1]*ln_Tr_green+W_coef_temp_green[2]*SSA_green_temp+W_coef_temp_green[3]*AAE_temp+W_coef_temp_green[4]*ln_Tr_green*SSA_green_temp+W_coef_temp_green[5]*SSA_green_temp*AAE_temp+W_coef_temp_green[6]*ln_Tr_green*AAE_temp+W_coef_temp_green[7]*ln_Tr_green*AAE_temp*SSA_green_temp)

	make /N=(npnts)/O b_abs_blue_temp=batn_blue*(W_coef_temp_blue[0]+W_coef_temp_blue[1]*ln_Tr_blue+W_coef_temp_blue[2]*SSA_blue_temp+W_coef_temp_blue[3]*AAE_temp+W_coef_temp_blue[4]*ln_Tr_blue*SSA_blue_temp+W_coef_temp_blue[5]*SSA_blue_temp*AAE_temp+W_coef_temp_blue[6]*ln_Tr_blue*AAE_temp+W_coef_temp_blue[7]*ln_Tr_blue*AAE_temp*SSA_blue_temp)

	//create new g_term, AAE and SSA
	g_term_red_temp=b_abs_red_temp/batn_red
	g_term_green_temp=b_abs_green_temp/batn_green
	g_term_blue_temp=b_abs_blue_temp/batn_blue

	AAE_power_fit(b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp)
	AAE_temp=AAE_three_lambda
 
	SSA_red_temp=bscat_red/(bscat_red+b_abs_red_temp)
	SSA_green_temp=bscat_green/(bscat_green+b_abs_green_temp)
	SSA_blue_temp=bscat_blue/(bscat_blue+b_abs_blue_temp)

	
	//check if convergence is reached (tolerance is set to be 1.5%)
	CurveFit/Q/ODR=2 line AAE_temp /x=AAE_previous_cycle /D 


while (W_coef[1]>1.015||W_coef[1]<0.985)



//Output final results
duplicate/O AAE_temp,AAE_final

duplicate/O  SSA_red_temp,SSA_red_final
duplicate/O  SSA_green_temp,SSA_green_final
duplicate/O  SSA_blue_temp,SSA_blue_final

duplicate/O b_abs_red_temp,b_abs_red_final
duplicate/O b_abs_green_temp,b_abs_green_final
duplicate/O b_abs_blue_temp,b_abs_blue_final

KillWaves g_term_red_temp,g_term_green_temp,g_term_blue_temp,SSA_red_temp,SSA_green_temp,SSA_blue_temp,b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp,AAE_previous_cycle

edit W_coef_initial_guess_all_wvl

end



function Algo_B()

wave bscat_red,bscat_green,bscat_blue,batn_red,batn_green,batn_blue,Tr_red,Tr_green,Tr_blue

variable npnts=numpnts(batn_red)


//initialize AAE, using batn
AAE_power_fit(batn_red,batn_green,batn_blue)
wave AAE_three_lambda
make /N=(npnts)/O AAE_temp=AAE_three_lambda  //"_temp" means that the term is a temporary variable, which is updated in the following loop 

//initialize SSA, using batn and bscat_ref
make /N=(npnts)/O SSA_red_temp=bscat_red/(bscat_red+batn_red)
make /N=(npnts)/O SSA_green_temp=bscat_green/(bscat_green+batn_green)
make /N=(npnts)/O SSA_blue_temp=bscat_blue/(bscat_blue+batn_blue)

//compute ln_Tr
make /N=(npnts)/O ln_Tr_red=ln(Tr_red)
make /N=(npnts)/O ln_Tr_green=ln(Tr_green)
make /N=(npnts)/O ln_Tr_blue=ln(Tr_blue)

//Initialize set of coefficients. FIREX_CLAP coefficients are used in the current code.

//Recommended initial guesses for the other combinations of 
//aerosol type and filter-based absorption photometer can be found in the manuscript.

Make/D/N=8/O  W_coef_temp_red= {0.27,	-0.16,	-0.18,	-0.05,	0.18,	0.08,	0.01,	0.03}
Make/D/N=8/O  W_coef_temp_green= {0.30,	-0.28,	-0.18,	-0.07,	0.25,	0.10,	0.12,	-0.17}
Make/D/N=8/O  W_coef_temp_blue= {0.32,	-0.38,	-0.19,	-0.08,	0.33,	0.12,	0.23,	-0.31}


//Calculate temporary Babs for each wavelength using Eq. (8) in the manuscript
make /n=(npnts)/o b_abs_red_temp=batn_red*(W_coef_temp_red[0]+W_coef_temp_red[1]*ln_Tr_red+W_coef_temp_red[2]*SSA_red_temp+W_coef_temp_red[3]*AAE_temp+W_coef_temp_red[4]*ln_Tr_red*SSA_red_temp+W_coef_temp_red[5]*SSA_red_temp*AAE_temp+W_coef_temp_red[6]*ln_Tr_red*AAE_temp+W_coef_temp_red[7]*ln_Tr_red*AAE_temp*SSA_red_temp)
make /n=(npnts)/o b_abs_green_temp=batn_green*(W_coef_temp_green[0]+W_coef_temp_green[1]*ln_Tr_green+W_coef_temp_green[2]*SSA_green_temp+W_coef_temp_green[3]*AAE_temp+W_coef_temp_green[4]*ln_Tr_green*SSA_green_temp+W_coef_temp_green[5]*SSA_green_temp*AAE_temp+W_coef_temp_green[6]*ln_Tr_green*AAE_temp+W_coef_temp_green[7]*ln_Tr_green*AAE_temp*SSA_green_temp)
make /n=(npnts)/o b_abs_blue_temp=batn_blue*(W_coef_temp_blue[0]+W_coef_temp_blue[1]*ln_Tr_blue+W_coef_temp_blue[2]*SSA_blue_temp+W_coef_temp_blue[3]*AAE_temp+W_coef_temp_blue[4]*ln_Tr_blue*SSA_blue_temp+W_coef_temp_blue[5]*SSA_blue_temp*AAE_temp+W_coef_temp_blue[6]*ln_Tr_blue*AAE_temp+W_coef_temp_blue[7]*ln_Tr_blue*AAE_temp*SSA_blue_temp)


//compute "g" term in Eq. (9), where g(Tr,SSA,AAE)=babs_ref/batn
make /N=(npnts)/O g_term_red_temp=b_abs_red_temp/batn_red
make /N=(npnts)/O g_term_green_temp=b_abs_green_temp/batn_green
make /N=(npnts)/O g_term_blue_temp=b_abs_blue_temp/batn_blue


variable loop_number=0
print "////////////iterative process////////////"  //show major results in the command window


do
	loop_number+=1
	print "Loop No.", loop_number

	//temporarily save the outputs from the previous iteration
	duplicate /O AAE_temp,AAE_previous_cycle  

	//implement "Algorithm A" on each wavelength
	make/D/N=8/O W_coef=W_coef_temp_red
	FuncFit/Q Algorithm_function W_coef g_term_red_temp /X={ln_Tr_red,SSA_red_temp,AAE_temp}  /w=batn_red  
		W_coef_temp_red=W_coef

	make/D/N=8/O W_coef =W_coef_temp_green
	FuncFit/Q Algorithm_function W_coef g_term_green_temp /X={ln_Tr_green,SSA_green_temp,AAE_temp}  /w=batn_green 
		W_coef_temp_green=W_coef

	make/D/N=8/O W_coef =W_coef_temp_blue
	FuncFit/Q Algorithm_function W_coef g_term_blue_temp /X={ln_Tr_blue,SSA_blue_temp,AAE_temp}  /w=batn_blue 
 		W_coef_temp_blue=W_coef


	//Update Babs for each wavelength using Eq. (8) in the manuscript
	b_abs_red_temp=batn_red*(W_coef_temp_red[0]+W_coef_temp_red[1]*ln_Tr_red+W_coef_temp_red[2]*SSA_red_temp+W_coef_temp_red[3]*AAE_temp+W_coef_temp_red[4]*ln_Tr_red*SSA_red_temp+W_coef_temp_red[5]*SSA_red_temp*AAE_temp+W_coef_temp_red[6]*ln_Tr_red*AAE_temp+W_coef_temp_red[7]*ln_Tr_red*AAE_temp*SSA_red_temp)

	b_abs_green_temp=batn_green*(W_coef_temp_green[0]+W_coef_temp_green[1]*ln_Tr_green+W_coef_temp_green[2]*SSA_green_temp+W_coef_temp_green[3]*AAE_temp+W_coef_temp_green[4]*ln_Tr_green*SSA_green_temp+W_coef_temp_green[5]*SSA_green_temp*AAE_temp+W_coef_temp_green[6]*ln_Tr_green*AAE_temp+W_coef_temp_green[7]*ln_Tr_green*AAE_temp*SSA_green_temp)

	b_abs_blue_temp=batn_blue*(W_coef_temp_blue[0]+W_coef_temp_blue[1]*ln_Tr_blue+W_coef_temp_blue[2]*SSA_blue_temp+W_coef_temp_blue[3]*AAE_temp+W_coef_temp_blue[4]*ln_Tr_blue*SSA_blue_temp+W_coef_temp_blue[5]*SSA_blue_temp*AAE_temp+W_coef_temp_blue[6]*ln_Tr_blue*AAE_temp+W_coef_temp_blue[7]*ln_Tr_blue*AAE_temp*SSA_blue_temp)

	//create new g_term, AAE and SSA
	g_term_red_temp=b_abs_red_temp/batn_red
	g_term_green_temp=b_abs_green_temp/batn_green
	g_term_blue_temp=b_abs_blue_temp/batn_blue

	AAE_power_fit(b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp)
	AAE_temp=AAE_three_lambda
 
	SSA_red_temp=bscat_red/(bscat_red+b_abs_red_temp)
	SSA_green_temp=bscat_green/(bscat_green+b_abs_green_temp)
	SSA_blue_temp=bscat_blue/(bscat_blue+b_abs_blue_temp)

	
	//check if convergence is reached (tolerance is set to be 1.5%)
	CurveFit/Q/ODR=2 line AAE_temp /x=AAE_previous_cycle /D 


while (W_coef[1]>1.015||W_coef[1]<0.985)



//Output the final results
duplicate/O AAE_temp,AAE_final

duplicate/O  SSA_red_temp,SSA_red_final
duplicate/O  SSA_green_temp,SSA_green_final
duplicate/O  SSA_blue_temp,SSA_blue_final

duplicate/O b_abs_red_temp,b_abs_red_final
duplicate/O b_abs_green_temp,b_abs_green_final
duplicate/O b_abs_blue_temp,b_abs_blue_final

KillWaves g_term_red_temp,g_term_green_temp,g_term_blue_temp,SSA_red_temp,SSA_green_temp,SSA_blue_temp,b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp,AAE_previous_cycle

end


function Algo_C()

wave batn_red,batn_green,batn_blue,Tr_red,Tr_green,Tr_blue

variable npnts=numpnts(batn_red)


//initialize AAE, using batn
AAE_power_fit(batn_red,batn_green,batn_blue)
wave AAE_three_lambda
make /N=(npnts)/O AAE_temp=AAE_three_lambda  //"_temp" means that the term is a temporary variable, which is updated in the following loop 

//initialize SSA, using AAE-SSA relationship 
variable a_652=1.19,b_652=1.49,c_652=1.84
variable a_528=1.16,b_528=1.44,c_528=1.68
variable a_467=1.13,b_467=1.41,c_467=1.59

make /n=(npnts)/o SSA_red_temp=((AAE_temp-a_652)/b_652)^(1/c_652)
make /n=(npnts)/o SSA_green_temp=((AAE_temp-a_528)/b_528)^(1/c_528)
make /n=(npnts)/o SSA_blue_temp=((AAE_temp-a_467)/b_467)^(1/c_467)

//compute bscat using SSA and Batn
make /n=(npnts)/o bscat_red=SSA_red_temp/(1-SSA_red_temp)*batn_red
make /n=(npnts)/o bscat_green=SSA_green_temp/(1-SSA_green_temp)*batn_green
make /n=(npnts)/o bscat_blue=SSA_blue_temp/(1-SSA_blue_temp)*batn_blue

//compute ln_Tr
make /N=(npnts)/O ln_Tr_red=ln(Tr_red)
make /N=(npnts)/O ln_Tr_green=ln(Tr_green)
make /N=(npnts)/O ln_Tr_blue=ln(Tr_blue)

//Initialize a set of coefficients. FIREX_CLAP coefficients are used in the current code.

//Recommended initial guesses for the other combinations of 
//aerosol type and filter-based absorption photometer can be found in the manuscript.

Make/D/N=8/O  W_coef_temp_red= {0.27,	-0.16,	-0.18,	-0.05,	0.18,	0.08,	0.01,	0.03}
Make/D/N=8/O  W_coef_temp_green= {0.30,	-0.28,	-0.18,	-0.07,	0.25,	0.10,	0.12,	-0.17}
Make/D/N=8/O  W_coef_temp_blue= {0.32,	-0.38,	-0.19,	-0.08,	0.33,	0.12,	0.23,	-0.31}

//Calculate temporary Babs for each wavelength using Eq. (8) in the manuscript
make /n=(npnts)/o b_abs_red_temp=batn_red*(W_coef_temp_red[0]+W_coef_temp_red[1]*ln_Tr_red+W_coef_temp_red[2]*SSA_red_temp+W_coef_temp_red[3]*AAE_temp+W_coef_temp_red[4]*ln_Tr_red*SSA_red_temp+W_coef_temp_red[5]*SSA_red_temp*AAE_temp+W_coef_temp_red[6]*ln_Tr_red*AAE_temp+W_coef_temp_red[7]*ln_Tr_red*AAE_temp*SSA_red_temp)
make /n=(npnts)/o b_abs_green_temp=batn_green*(W_coef_temp_green[0]+W_coef_temp_green[1]*ln_Tr_green+W_coef_temp_green[2]*SSA_green_temp+W_coef_temp_green[3]*AAE_temp+W_coef_temp_green[4]*ln_Tr_green*SSA_green_temp+W_coef_temp_green[5]*SSA_green_temp*AAE_temp+W_coef_temp_green[6]*ln_Tr_green*AAE_temp+W_coef_temp_green[7]*ln_Tr_green*AAE_temp*SSA_green_temp)
make /n=(npnts)/o b_abs_blue_temp=batn_blue*(W_coef_temp_blue[0]+W_coef_temp_blue[1]*ln_Tr_blue+W_coef_temp_blue[2]*SSA_blue_temp+W_coef_temp_blue[3]*AAE_temp+W_coef_temp_blue[4]*ln_Tr_blue*SSA_blue_temp+W_coef_temp_blue[5]*SSA_blue_temp*AAE_temp+W_coef_temp_blue[6]*ln_Tr_blue*AAE_temp+W_coef_temp_blue[7]*ln_Tr_blue*AAE_temp*SSA_blue_temp)


//compute "g" term in Eq. (9), where g(Tr,SSA,AAE)=babs_ref/batn
make /N=(npnts)/O g_term_red_temp=b_abs_red_temp/batn_red
make /N=(npnts)/O g_term_green_temp=b_abs_green_temp/batn_green
make /N=(npnts)/O g_term_blue_temp=b_abs_blue_temp/batn_blue



variable loop_number=0
print "////////////iterative process////////////"  //show major results in the command window


do
	loop_number+=1
	print "Loop No.", loop_number

	//temporarily save the outputs from the previous iteration
	duplicate /O AAE_temp,AAE_previous_cycle  

	//implement "Algorithm A" on each wavelength
	make/D/N=8/O W_coef=W_coef_temp_red
	FuncFit/Q Algorithm_function W_coef g_term_red_temp /X={ln_Tr_red,SSA_red_temp,AAE_temp}  /w=batn_red  
		W_coef_temp_red=W_coef

	make/D/N=8/O W_coef =W_coef_temp_green
	FuncFit/Q Algorithm_function W_coef g_term_green_temp /X={ln_Tr_green,SSA_green_temp,AAE_temp}  /w=batn_green 
		W_coef_temp_green=W_coef
		
	make/D/N=8/O W_coef =W_coef_temp_blue
	FuncFit/Q Algorithm_function W_coef g_term_blue_temp /X={ln_Tr_blue,SSA_blue_temp,AAE_temp}  /w=batn_blue 
 		W_coef_temp_blue=W_coef


	//Update Babs for each wavelength using Eq. (8) in the manuscript
	b_abs_red_temp=batn_red*(W_coef_temp_red[0]+W_coef_temp_red[1]*ln_Tr_red+W_coef_temp_red[2]*SSA_red_temp+W_coef_temp_red[3]*AAE_temp+W_coef_temp_red[4]*ln_Tr_red*SSA_red_temp+W_coef_temp_red[5]*SSA_red_temp*AAE_temp+W_coef_temp_red[6]*ln_Tr_red*AAE_temp+W_coef_temp_red[7]*ln_Tr_red*AAE_temp*SSA_red_temp)

	b_abs_green_temp=batn_green*(W_coef_temp_green[0]+W_coef_temp_green[1]*ln_Tr_green+W_coef_temp_green[2]*SSA_green_temp+W_coef_temp_green[3]*AAE_temp+W_coef_temp_green[4]*ln_Tr_green*SSA_green_temp+W_coef_temp_green[5]*SSA_green_temp*AAE_temp+W_coef_temp_green[6]*ln_Tr_green*AAE_temp+W_coef_temp_green[7]*ln_Tr_green*AAE_temp*SSA_green_temp)

	b_abs_blue_temp=batn_blue*(W_coef_temp_blue[0]+W_coef_temp_blue[1]*ln_Tr_blue+W_coef_temp_blue[2]*SSA_blue_temp+W_coef_temp_blue[3]*AAE_temp+W_coef_temp_blue[4]*ln_Tr_blue*SSA_blue_temp+W_coef_temp_blue[5]*SSA_blue_temp*AAE_temp+W_coef_temp_blue[6]*ln_Tr_blue*AAE_temp+W_coef_temp_blue[7]*ln_Tr_blue*AAE_temp*SSA_blue_temp)

	//create new g_term, AAE and SSA
	g_term_red_temp=b_abs_red_temp/batn_red
	g_term_green_temp=b_abs_green_temp/batn_green
	g_term_blue_temp=b_abs_blue_temp/batn_blue

	AAE_power_fit(b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp)
	AAE_temp=AAE_three_lambda
 
	SSA_red_temp=bscat_red/(bscat_red+b_abs_red_temp)
	SSA_green_temp=bscat_green/(bscat_green+b_abs_green_temp)
	SSA_blue_temp=bscat_blue/(bscat_blue+b_abs_blue_temp)

	
	//check if convergence is reached (tolerance is set to be 1.5%)
	CurveFit/Q/ODR=2 line AAE_temp /x=AAE_previous_cycle /D 


while (W_coef[1]>1.015||W_coef[1]<0.985)



//Output the final results
duplicate/O AAE_temp,AAE_final

duplicate/O  SSA_red_temp,SSA_red_final
duplicate/O  SSA_green_temp,SSA_green_final
duplicate/O  SSA_blue_temp,SSA_blue_final

duplicate/O b_abs_red_temp,b_abs_red_final
duplicate/O b_abs_green_temp,b_abs_green_final
duplicate/O b_abs_blue_temp,b_abs_blue_final

KillWaves g_term_red_temp,g_term_green_temp,g_term_blue_temp,SSA_red_temp,SSA_green_temp,SSA_blue_temp,b_abs_red_temp,b_abs_green_temp,b_abs_blue_temp,AAE_previous_cycle

end


function AAE_power_fit(babs_wvl_red,babs_wvl_green,babs_wvl_blue)
wave babs_wvl_red,babs_wvl_green,babs_wvl_blue


//the operated wavelengths are specified by the manufacture
make /O/N=3 wavelength_instrument={652,528,467}


variable npnts=numpnts(babs_wvl_red)

//compute AAE_two_lambda, which are used later when initializing AAE_three_lambda
make /N=(npnts)/O AAE_R_G=-ln(babs_wvl_red/babs_wvl_green)/ln(wavelength_instrument[0]/wavelength_instrument[1])
make /N=(npnts)/O AAE_R_B=-ln(babs_wvl_red/babs_wvl_blue)/ln(wavelength_instrument[0]/wavelength_instrument[2])
make /N=(npnts)/O AAE_G_B=-ln(babs_wvl_green/babs_wvl_blue)/ln(wavelength_instrument[1]/wavelength_instrument[2])


make /N=(npnts)/O AAE_three_lambda=nan


//initialize coefficients in babs-AAE function
variable temp_AAE,temp_AAE_constant,temp_sigma,temp_sigma_constant

variable m
for (m=0;m<(npnts);m+=1)

if (numtype(babs_wvl_red[m])==0&&numtype(babs_wvl_green[m])==0&&numtype(babs_wvl_blue[m])==0)

	make /O /N=3 one_babs_at_three_lambda={babs_wvl_red[m],babs_wvl_green[m],babs_wvl_blue[m]}
		
	////use AAE_G_B as initial input
	make/D/N=2/O W_coef = {AAE_G_B[m],babs_wvl_green[m]/ (528^(-AAE_G_B[m])) }  
	
	FuncFit/Q AAE_fit_one_measurement W_coef one_babs_at_three_lambda /X=wavelength_instrument /D 
	wave W_coef,w_sigma
	temp_AAE=W_coef[0];temp_AAE_constant=W_coef[1];temp_sigma=w_sigma[0];temp_sigma_constant=w_sigma[1]

	////use AAE_R_G as initial input
	make/D/N=2/O W_coef = {AAE_R_G[m],babs_wvl_red[m]/ (652^(-AAE_R_G[m]))	}  

	FuncFit/Q AAE_fit_one_measurement W_coef one_babs_at_three_lambda /X=wavelength_instrument /D 

	//check if the current inital guess (AAE_R_G) is better than the previous one (AAE_G_B): ? w_sigma is smaller
	if (w_sigma[0]<temp_sigma)  //Yes, the current w_sigma is smaller
		
		temp_AAE=W_coef[0];temp_AAE_constant=W_coef[1];temp_sigma=w_sigma[0];temp_sigma_constant=w_sigma[1]	
	
	else 
		
		//do not replace the previous results
	
	endif
			
	////use AAE_R_B as initial input
	make/D/N=2/O W_coef = {AAE_R_B[m],babs_wvl_blue[m]/ (467^(-AAE_R_B[m]))	}  

	FuncFit/Q AAE_fit_one_measurement W_coef one_babs_at_three_lambda  /X=wavelength_instrument /D 
		
	//check if the current initial guess (AAE_R_B) is better than the previous one (AAE_G_B or AAE_R_G): ? w_sigma is smaller
	if (w_sigma[0]<temp_sigma)  //Yes, the current w_sigma is smaller
				
		temp_AAE=W_coef[0];temp_AAE_constant=W_coef[1];temp_sigma=w_sigma[0];temp_sigma_constant=w_sigma[1]		

	else 
		//do not replace the previous results
	endif	
				

//Output the results		
	AAE_three_lambda[m]=temp_AAE


	temp_AAE=nan;temp_AAE_constant=nan;temp_AAE=nan;temp_sigma=nan

endif
endfor


end




Function AAE_fit_one_measurement(w,wavelength) : FitFunc
	Wave w
	Variable wavelength

	//CurveFitDialog/ These comments were created by the Curve Fitting dialog. Altering them will
	//CurveFitDialog/ make the function less convenient to work with in the Curve Fitting dialog.
	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(wavelength) = AAE_constant*wavelength^(-AAE_cal)
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ wavelength
	//CurveFitDialog/ Coefficients 2
	//CurveFitDialog/ w[0] = AAE_cal
	//CurveFitDialog/ w[1] = AAE_constant

	return w[1]*wavelength^(-w[0])
End




Function Algorithm_function(w,ln_tr_in_function,SSA_in_function, AAE_in_function) : FitFunc
	Wave w
	Variable ln_tr_in_function
	Variable SSA_in_function
	Variable AAE_in_function

	//CurveFitDialog/ These comments were created by the Curve Fitting dialog. Altering them will
	//CurveFitDialog/ make the function less convenient to work with in the Curve Fitting dialog.
	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(ln_tr_in_function,SSA_in_function,aae_in_function) = c0+c1*ln_tr_in_function+c2*SSA_in_function+c3*aae_in_function+c4*ln_tr_in_function*SSA_in_function+c5*SSA_in_function*aae_in_function+c6*ln_tr_in_function*aae_in_function+c7*ln_tr_in_function*aae_in_function*SSA_in_function
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 3
	//CurveFitDialog/ SSA_in_function
	//CurveFitDialog/ aae_in_function
	//CurveFitDialog/ ln_tr_in_function
	//CurveFitDialog/ Coefficients 8
	//CurveFitDialog/ w[0] = c0
	//CurveFitDialog/ w[1] = c1
	//CurveFitDialog/ w[2] = c2
	//CurveFitDialog/ w[3] = c3
	//CurveFitDialog/ w[4] = c4
	//CurveFitDialog/ w[5] = c5
	//CurveFitDialog/ w[6] = C6
	//CurveFitDialog/ w[7] = C7
	return w[0]+w[1]*ln_tr_in_function+w[2]*SSA_in_function+w[3]*aae_in_function+w[4]*ln_tr_in_function*SSA_in_function+w[5]*SSA_in_function*aae_in_function+w[6]*ln_tr_in_function*aae_in_function+W[7]*ln_tr_in_function*aae_in_function*SSA_in_function
End


Function Decimal_control(value, numSigDigits)
variable value, numSigDigits

string str
 
sprintf str, "%.*g\r", numSigDigits, value
 
return str2num(str)

End

