#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Tue Jul 16 14:24:24 2019

@author: zhangshuai
"""

import core

import os
import csv
import math
import datetime
import numpy as np
from numba import njit
import matplotlib as mpl
import matplotlib.pyplot as plt
import matplotlib.colors as col

@njit
def running_calculation(data,func,size=13,mask=-999.): #1km
    row,col=data.shape
    result=np.full_like(data,mask)
    for i in range(row):
        for j in range(col):
            if data[i,j]!=mask:
                buf=[]
                
                for k in range(-(size//2),size//2+1):
                    if (j+k)<0 or (j+k)>=col:
                        continue
                    else:
                        if data[i,j+k]!=mask:
                            buf.append(data[i,j+k])
                            
                if len(buf)!=1:
                    if func==0: #std
                        result[i,j]=np.std(np.array(buf))
                    elif func==1: #mean
                        result[i,j]=np.mean(np.array(buf))
                    
    return result

def cappi_calculation(radar,altitude,radar_altitude=87/1000,radar_resolution=.075):
    a=8500
    
    result=np.full_like(radar.data[(1,'Z')],-999.)
    
    sweep_altitude=[]
    Range=np.arange(1,len(result[0])+1)*radar_resolution
    for sweep in range(1,radar.sweepnum+1):
        elevation=core.elevation[sweep]
        sweep_altitude.append(np.sqrt(Range**2+a**2+2*a*Range*np.sin(math.radians(elevation)))-a+radar_altitude)
    sweep_altitude=np.array(sweep_altitude)
    
    start_index=0
    end_index=len(result[0])
    for index,value in enumerate(sweep_altitude[radar.sweepnum-1]):
        if value>altitude:
            start_index=index-1
            break
    for index,value in enumerate(sweep_altitude[0]):
        if value>altitude:
            end_index=index+1
            break
    
    sweep_altitude=np.fabs(sweep_altitude-altitude)
    sweep_index=np.argmin(sweep_altitude,axis=0)+1
    sweep_index[:start_index]=0
    sweep_index[end_index:]=0
    
    for sweep in range(1,radar.sweepnum+1):
        result[:,sweep_index==sweep]=radar.data[(sweep,'Z')][:,sweep_index[:len(radar.data[(sweep,'Z')][0])]==sweep]
        
    return result

def cappi_visualization(radar,info):
    data,altitude,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath=info
    
    data=np.vstack((data,data[0]))
    data=np.ma.masked_values(data,-999.)
    azimuth=radar.azimuth[1]*np.pi/180
    Range=np.arange(1,len(data[0])+1)*resolution
    Range,azimuth=np.meshgrid(Range,azimuth)
    azimuth=np.vstack((azimuth,azimuth[0]))
    Range=np.vstack((Range,Range[0]))
    
    plt.close('all')
    fig,axe=plt.subplots(subplot_kw=dict(polar=True))
    
    norm=mpl.colors.Normalize(vmin=norm[0],vmax=norm[1])
    cmap=plt.get_cmap(cmap)
        
    axe.set_theta_direction(-1)
    axe.set_theta_zero_location('N')
    axe.set_thetagrids(list(range(0,360,30)),visible=False)
    axe.set_rgrids(list(range(rangecircle,maxrange,rangecircle))+[maxrange],visible=True,color='#b0b0b0')
    pcm=axe.pcolormesh(azimuth,Range,data,norm=norm,cmap=cmap)
    cb=fig.colorbar(pcm,ax=axe)
    cb.set_label(cblabel)
    axe.set_rmax(maxrange)
    axe.grid(True)
    axe.set_facecolor('#000000')
                      
    axe.set_title('time: '+str(datetime.datetime.utcfromtimestamp(radar.time))+' [UTC]\n'+\
                  'altitude: '+str(altitude)+' [km]')
    fig.tight_layout()
    if figpath!=None:
        ID='NUIST'
        time=str(datetime.datetime.utcfromtimestamp(radar.time))
        time=time.replace('-','')
        time=time.replace(':','')
        time=time.replace(' ','')
        version=radar.version
        extension='png'
        figname='{}.{}.{}.{}.{}'.format(ID,time,'cappi',version,extension)
        plt.savefig(os.path.join(figpath,figname),dpi=dpi)
    else:
        plt.show()

class METSIGNAL:
    def __init__(self,radar):
        self.radar=radar
    
    def feature_calculation(self):
        self.feature={}
        
        for sweep in range(1,self.radar.sweepnum+1):
            z=self.radar.data[(sweep,'Z')]
            v=self.radar.data[(sweep,'V')]
            zdr=self.radar.data[(sweep,'ZDR')]
            cc=self.radar.data[(sweep,'CC')]
            
            sd_zdr=running_calculation(zdr,0)
            sd_cc=running_calculation(cc,0)
            
            self.feature[sweep]=(z,v,cc,sd_zdr,sd_cc)
            
#            break
        
    def feature_visualization(self,info):
        index2moment={0:'Z',1:'V',2:'CC',3:'SD(ZDR)',4:'SD(CC)'}
        
        sweep,moment_index,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath=info
        
        data=self.feature[sweep][moment_index]
        data=np.vstack((data,data[0]))
        data=np.ma.masked_values(data,-999.)
        azimuth=self.radar.azimuth[sweep]*np.pi/180
        Range=np.arange(1,len(data[0])+1)*resolution
        Range,azimuth=np.meshgrid(Range,azimuth)
        azimuth=np.vstack((azimuth,azimuth[0]))
        Range=np.vstack((Range,Range[0]))
        
        plt.close('all')
        fig,axe=plt.subplots(subplot_kw=dict(polar=True))
        
        norm=mpl.colors.Normalize(vmin=norm[0],vmax=norm[1])
        cmap=plt.get_cmap(cmap)
            
        axe.set_theta_direction(-1)
        axe.set_theta_zero_location('N')
        axe.set_thetagrids(list(range(0,360,30)),visible=False)
        axe.set_rgrids(list(range(rangecircle,maxrange,rangecircle))+[maxrange],visible=True,color='#b0b0b0')
        pcm=axe.pcolormesh(azimuth,Range,data,norm=norm,cmap=cmap)
        cb=fig.colorbar(pcm,ax=axe)
        cb.set_label(cblabel)
        axe.set_rmax(maxrange)
        axe.grid(True)
        axe.set_facecolor('#000000')
                          
        axe.set_title('time: '+str(datetime.datetime.utcfromtimestamp(self.radar.time))+' [UTC]\n'+\
                      'elevation: '+str(np.around(core.elevation[sweep],decimals=1))+' [deg]')
        fig.tight_layout()
        if figpath!=None:
            ID='NUIST'
            time=str(datetime.datetime.utcfromtimestamp(self.radar.time))
            time=time.replace('-','')
            time=time.replace(':','')
            time=time.replace(' ','')
            version=self.radar.version
            extension='png'
            figname='{}.{}.{}.{}.{}.{}'.format(ID,time,sweep,index2moment[moment_index],version,extension)
            plt.savefig(os.path.join(figpath,figname),dpi=dpi)
        else:
            plt.show()
            
    def fuzzy_logic(self):
        feature2weight={'Z':1,'V':1,'CC':1,'SD(ZDR)':2,'SD(CC)':1}
        feature2index={'Z':0,'V':1,'CC':2,'SD(ZDR)':3,'SD(CC)':4}
        feature2inflexion={'Z':(-5,20,30,55),'V':(-5,-1,1,5),'CC':(.5,.9),'SD(ZDR)':(1,2),'SD(CC)':(.05,.15)}
        feature2mfshape={'Z':'updown','V':'downup','CC':'up','SD(ZDR)':'down','SD(CC)':'down'}
        
        self.A={}
        
        for sweep in range(1,self.radar.sweepnum+1):
            A=np.zeros_like(self.feature[sweep][0])
            denominator=np.zeros_like(self.feature[sweep][0])
            
            for feature in ('Z','V','CC','SD(ZDR)','SD(CC)'):
                Bool=self.feature[sweep][feature2index[feature]]!=-999
                A[Bool]+=feature2weight[feature]*trapezoidfunc(self.feature[sweep][feature2index[feature]][Bool],feature2inflexion[feature],feature2mfshape[feature])
                denominator[Bool]+=feature2weight[feature]
                
            Bool=denominator!=0
            A[Bool]/=denominator[Bool]
            Bool=self.radar.data[(sweep,'Z')]==-999
            A[Bool]=-999
            
            self.A[sweep]=A
            
#            break
            
    def A_visualization(self,info):
        sweep,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath=info
        
        data=self.A[sweep]
        data=np.vstack((data,data[0]))
        data=np.ma.masked_values(data,-999.)
        azimuth=self.radar.azimuth[sweep]*np.pi/180
        Range=np.arange(1,len(data[0])+1)*resolution
        Range,azimuth=np.meshgrid(Range,azimuth)
        azimuth=np.vstack((azimuth,azimuth[0]))
        Range=np.vstack((Range,Range[0]))
        
        plt.close('all')
        fig,axe=plt.subplots(subplot_kw=dict(polar=True))
        
        norm=mpl.colors.Normalize(vmin=norm[0],vmax=norm[1])
        cmap=plt.get_cmap(cmap)
            
        axe.set_theta_direction(-1)
        axe.set_theta_zero_location('N')
        axe.set_thetagrids(list(range(0,360,30)),visible=False)
        axe.set_rgrids(list(range(rangecircle,maxrange,rangecircle))+[maxrange],visible=True,color='#b0b0b0')
        pcm=axe.pcolormesh(azimuth,Range,data,norm=norm,cmap=cmap)
        cb=fig.colorbar(pcm,ax=axe)
        cb.set_label(cblabel)
        axe.set_rmax(maxrange)
        axe.grid(True)
        axe.set_facecolor('#000000')
                          
        axe.set_title('time: '+str(datetime.datetime.utcfromtimestamp(self.radar.time))+' [UTC]\n'+\
                      'elevation: '+str(np.around(core.elevation[sweep],decimals=1))+' [deg]')
        fig.tight_layout()
        if figpath!=None:
            ID='NUIST'
            time=str(datetime.datetime.utcfromtimestamp(self.radar.time))
            time=time.replace('-','')
            time=time.replace(':','')
            time=time.replace(' ','')
            version=self.radar.version
            extension='png'
            figname='{}.{}.{}.{}.{}.{}'.format(ID,time,sweep,'A',version,extension)
            plt.savefig(os.path.join(figpath,figname),dpi=dpi)
        else:
            plt.show()
            
    def post_processing(self,zdr_thresh=4.5,cc_thresh=.65,cappi_thresh=11):
        cappi=cappi_calculation(self.radar,3)
        
        for sweep in range(1,self.radar.sweepnum+1):
            zdr=self.radar.data[(sweep,'ZDR')]
            cc=self.radar.data[(sweep,'CC')]
            z=self.radar.data[(sweep,'Z')]
            
            Bool=((zdr>zdr_thresh)|(zdr<-zdr_thresh)|(cc<cc_thresh))&(z!=-999)
            self.A[sweep][Bool]=0
            
            col=len(self.A[sweep][0])
            Bool=(cappi[:,:col]>cappi_thresh)&(z!=-999)
            self.A[sweep][Bool]=1
        
#            break
        
    def get_result(self):
        month=int(datetime.datetime.utcfromtimestamp(self.radar.time).month)
        if 4<=month<10: #summer
            thresh=.8
        else: #winter
            thresh=.7
        
        self.result={}
        
        for sweep in range(1,self.radar.sweepnum+1):
            self.result[sweep]=np.full_like(self.A[sweep],-999.)
            Bool=(self.A[sweep]!=-999)&(self.A[sweep]>=thresh)
            self.result[sweep][Bool]=1
            Bool=(self.A[sweep]!=-999)&(self.A[sweep]<thresh)
            self.result[sweep][Bool]=0
            
#            break
        
        self.radar.version=2 #after metsignal is 2
        
    def metsignal_visualization(self,info):
        sweep,cblabel,resolution,rangecircle,maxrange,dpi,figpath=info
        
        data=self.result[sweep]
        data=np.vstack((data,data[0]))
        data=np.ma.masked_values(data,-999.)
        azimuth=self.radar.azimuth[sweep]*np.pi/180
        Range=np.arange(1,len(data[0])+1)*resolution
        Range,azimuth=np.meshgrid(Range,azimuth)
        azimuth=np.vstack((azimuth,azimuth[0]))
        Range=np.vstack((Range,Range[0]))
        
        plt.close('all')
        fig,axe=plt.subplots(subplot_kw=dict(polar=True))
        
        cmap=mpl.colors.ListedColormap(['#ff7f0e','#1f77b4'])
        norm=col.BoundaryNorm(boundaries=[-.5,.5,1.5],ncolors=cmap.N)
            
        axe.set_theta_direction(-1)
        axe.set_theta_zero_location('N')
        axe.set_thetagrids(list(range(0,360,30)),visible=False)
        axe.set_rgrids(list(range(rangecircle,maxrange,rangecircle))+[maxrange],visible=True,color='#b0b0b0')
        pcm=axe.pcolormesh(azimuth,Range,data,norm=norm,cmap=cmap)
        cb=fig.colorbar(pcm,ax=axe)
        cb.set_label(cblabel)
        cb.set_ticks([0,1])
        cb.set_ticklabels(['NMET','MET'])
        axe.set_rmax(maxrange)
        axe.grid(True)
        axe.set_facecolor('#000000')
                          
        axe.set_title('time: '+str(datetime.datetime.utcfromtimestamp(self.radar.time))+' [UTC]\n'+\
                      'elevation: '+str(np.around(core.elevation[sweep],decimals=1))+' [deg]')
        fig.tight_layout()
        if figpath!=None:
            ID='NUIST'
            time=str(datetime.datetime.utcfromtimestamp(self.radar.time))
            time=time.replace('-','')
            time=time.replace(':','')
            time=time.replace(' ','')
            version=self.radar.version
            extension='png'
            figname='{}.{}.{}.{}.{}.{}'.format(ID,time,sweep,'metsignal',version,extension)
            plt.savefig(os.path.join(figpath,figname),dpi=dpi)
        else:
            plt.show()
                   
def train_table():
    filepath=r'/Volumes/dusten/nc'
    savepath=r'/Users/zhangshuai/Desktop/train_table.csv'
    filelist=os.listdir(filepath)
    filelist.sort()
    
    if '.DS_Store' in filelist:
        del filelist[filelist.index('.DS_Store')]
    
    with open(savepath, 'w', newline='') as f:
        writer=csv.writer(f)
        writer.writerow(['file','met','nmet','azi_start','azi_end','range_start','range_end'])
        for file in filelist:
            writer.writerow([file])
            
def get_train_set():
    filepath=r'/Volumes/dusten/nc'
    csvpath=r'/Users/zhangshuai/Desktop/NUIST-CDP/Part II/train_table.csv'
    savepath=r'/Users/zhangshuai/Desktop/NUIST-CDP/Part II/train_set'
    
    with open(csvpath,'r') as f:
        lines=csv.reader(f)
        for index,line in enumerate(lines):
            if index:
                if line[1]!='':
                    Type='met'
                    if len(line[1])!=1:
                        sweeps=line[1].split()
                        for i in range(len(sweeps)):
                            sweeps[i]=int(sweeps[i])
                    else:
                        sweeps=[int(line[1])]
                elif line[2]!='':
                    Type='nmet'
                    sweeps=[int(line[2])]
                else:
                    continue
                
                file=line[0]
                print(file)
                
                radar=core.nc2radar(os.path.join(filepath,file))
                metsignal=METSIGNAL(radar)
                metsignal.feature_calculation()
                
                row=len(metsignal.feature[1][0])
                azi_start=int(row*int(line[3])/360)
                azi_end=int(row*int(line[4])/360)
                range_start=int(int(line[5])/.075)
                range_end=int(int(line[6])/.075)
                
                for sweep in sweeps:
                    z=metsignal.feature[sweep][0][azi_start:azi_end,range_start:range_end]
                    v=metsignal.feature[sweep][1][azi_start:azi_end,range_start:range_end]
                    cc=metsignal.feature[sweep][2][azi_start:azi_end,range_start:range_end]
                    sd_zdr=metsignal.feature[sweep][3][azi_start:azi_end,range_start:range_end]
                    sd_cc=metsignal.feature[sweep][4][azi_start:azi_end,range_start:range_end]
                    
                    Bool=(z!=-999)
                    f0=z[Bool].reshape(-1,1)
                    f1=v[Bool].reshape(-1,1)
                    f2=cc[Bool].reshape(-1,1)
                    f3=sd_zdr[Bool].reshape(-1,1)
                    f4=sd_cc[Bool].reshape(-1,1)
                    f=np.hstack((f0,f1,f2,f3,f4))
                    
                    np.save(os.path.join(os.path.join(savepath,Type),file+str(sweep)+'.npy'),f)
                    
def distribution():
    feature2index={'Z':0,'V':1,'CC':2,'SD(ZDR)':3,'SD(CC)':4}
    feature2range={'Z':(-10,60),'V':(-13,13),'CC':(0,1),'SD(ZDR)':(0,9),'SD(CC)':(0,.35)}
    feature2xlabel={'Z':'Z [dBZ]','V':'V [m/s]','CC':'CC','SD(ZDR)':'SD(ZDR) [dB]','SD(CC)':'SD(CC)'}
    feature2inflexion={'Z':(0,20,30,50),'V':(-5,-1,1,5),'CC':(.5,.95),'SD(ZDR)':(1,2),'SD(CC)':(.05,.15)}
    feature2mfshape={'Z':'updown','V':'downup','CC':'up','SD(ZDR)':'down','SD(CC)':'down'}
    
    filepath=r'/Users/zhangshuai/Desktop/NUIST-CDP/Part II/train_set'
    for file in os.listdir(os.path.join(filepath,'nmet')):
        if file!='.DS_Store':
            if 'trainingset_nmet' not in vars():
                trainingset_nmet=np.load(os.path.join(os.path.join(filepath,'nmet'),file))
            else:
                trainingset_nmet=np.vstack((trainingset_nmet,np.load(os.path.join(os.path.join(filepath,'nmet'),file))))
    for file in os.listdir(os.path.join(filepath,'met')):
        if file!='.DS_Store':
            if 'trainingset_met' not in vars():
                trainingset_met=np.load(os.path.join(os.path.join(filepath,'met'),file))
            else:
                trainingset_met=np.vstack((trainingset_met,np.load(os.path.join(os.path.join(filepath,'met'),file))))
                        
    for feature in ('Z','V','CC','SD(ZDR)','SD(CC)'):
        plt.close('all')
        fig,ax=plt.subplots()
        
        met=trainingset_met[:,feature2index[feature]]
        met=met[met!=-999]
        nmet=trainingset_nmet[:,feature2index[feature]]
        nmet=nmet[nmet!=-999]
        pd_met,_,patches_met=ax.hist(met,50,density=True,alpha=0.5,label='MET',range=feature2range[feature])
        pd_nmet,_,patches_nmet=ax.hist(nmet,50,density=True,alpha=0.5,label='NMET',range=feature2range[feature])
        
        ax.set_xlabel(feature2xlabel[feature])
        ax.set_ylabel('probability density')
        ax.grid(True)
        if feature == 'V':
            ax.legend(handles=[patches_met[0],patches_nmet[0]],bbox_to_anchor=(1, 0.95))
        else:
            ax.legend(handles=[patches_met[0],patches_nmet[0]])
        plt.tight_layout()
#        plt.show()
        
        '''used to determin the mf'''
#        mf=pd_met/(pd_nmet+pd_met)
#        plt.plot(_[:-1],mf)
#        plt.grid(True)
#        plt.show()
        
        ax1=ax.twinx()
        start,end=feature2range[feature]
        x=np.linspace(start,end,100)
        y=trapezoidfunc(x,feature2inflexion[feature],feature2mfshape[feature])
        
        ax1.plot(x,y,color='red',linewidth=4.0)
        ax1.set_xlim(feature2range[feature])
        ax1.set_ylim(-0.05,1.05)
        ax1.set_ylabel('MF Value')
        plt.show()
#        plt.tight_layout()
#        plt.savefig(os.path.join(r'/Users/zhangshuai/Desktop',feature+'.png'),dpi=300)
        
    del trainingset_nmet
    del trainingset_met
    
def trapezoidfunc(Input,inflexion,mfshape):
    output=np.zeros_like(Input)
    
    if mfshape=='up':
        slope=1/(inflexion[1]-inflexion[0])
        intercept=-inflexion[0]*slope
        
        output[Input>=inflexion[1]]=1
        Bool=(Input<inflexion[1]) & (Input>inflexion[0])
        output[Bool]=Input[Bool]*slope+intercept
    elif mfshape=='down':
        slope=1/(inflexion[0]-inflexion[1])
        intercept=-inflexion[1]*slope
                          
        output[Input<=inflexion[0]]=1
        Bool=(Input<inflexion[1]) & (Input>inflexion[0])
        output[Bool]=Input[Bool]*slope+intercept
    elif mfshape=='updown':
        slope=1/(inflexion[1]-inflexion[0])
        intercept=-inflexion[0]*slope
        
        output[(Input>=inflexion[1]) & (Input<=inflexion[2])]=1
        Bool=(Input<inflexion[1]) & (Input>inflexion[0])
        output[Bool]=Input[Bool]*slope+intercept
        
        slope=1/(inflexion[2]-inflexion[3])
        intercept=-inflexion[3]*slope
                          
        Bool=(Input<inflexion[3]) & (Input>inflexion[2])
        output[Bool]=Input[Bool]*slope+intercept
    elif mfshape=='downup':
        slope=1/(inflexion[0]-inflexion[1])
        intercept=-inflexion[1]*slope
                          
        output[Input<=inflexion[0]]=1
        Bool=(Input<inflexion[1]) & (Input>inflexion[0])
        output[Bool]=Input[Bool]*slope+intercept
        
        slope=1/(inflexion[3]-inflexion[2])
        intercept=-inflexion[2]*slope
        
        output[Input>=inflexion[3]]=1
        Bool=(Input<inflexion[3]) & (Input>inflexion[2])
        output[Bool]=Input[Bool]*slope+intercept
                     
    return output

def test(mode):
    filepath=r'/Volumes/dusten/nc'
    csvpath=r'/Users/zhangshuai/Desktop/NUIST-CDP/Part II/test_table.csv'
    
    type2index={'met':1,'nmet':0}
    
    with open(csvpath,'r') as f:
        lines=csv.reader(f)
        
        nmet_total_num=0
        nmet_right_num=0
        met_total_num=0
        met_right_num=0
        all_total_num=0
        all_right_num=0
        
        for index,line in enumerate(lines):
            if index:
                if line[1]!='':
                    Type='met'
                    if len(line[1])!=1:
                        sweeps=line[1].split()
                        for i in range(len(sweeps)):
                            sweeps[i]=int(sweeps[i])
                    else:
                        sweeps=[int(line[1])]
                elif line[2]!='':
                    Type='nmet'
                    sweeps=[int(line[2])]
                else:
                    continue
                
                file=line[0]
                print(file)
                
                radar=core.nc2radar(os.path.join(filepath,file))
                metsignal=METSIGNAL(radar)
                metsignal.feature_calculation()
                metsignal.fuzzy_logic()
                metsignal.post_processing()
                metsignal.get_result()
                
                row=len(metsignal.result[1])
                azi_start=int(row*int(line[3])/360)
                azi_end=int(row*int(line[4])/360)
                range_start=int(int(line[5])/.075)
                range_end=int(int(line[6])/.075)
                
                for sweep in sweeps:
                    z=radar.data[sweep,'Z'][azi_start:azi_end,range_start:range_end]
                    snr=radar.data[sweep,'SNR'][azi_start:azi_end,range_start:range_end]
                    result=metsignal.result[sweep][azi_start:azi_end,range_start:range_end]
                    if mode==0:
                        Bool=(snr>=5)&(z!=-999)
                    elif mode==1:
                        Bool=(snr>=15)&(z!=-999)
                    elif mode==2:
                        Bool=(snr>=5)&(snr<15)&(z!=-999)
                    result=result[Bool]
                    total_num=len(result)
                    right_num=len(result[result==type2index[Type]])
                    
#                    print(right_num,total_num,right_num/total_num)
                    all_total_num+=total_num
                    all_right_num+=right_num
                    
                    if Type=='nmet':
                        nmet_total_num+=total_num
                        nmet_right_num+=right_num
                    else:
                        met_total_num+=total_num
                        met_right_num+=right_num
                    
#                break
            
    print(met_right_num,met_total_num,met_right_num/met_total_num)
    print(nmet_right_num,nmet_total_num,nmet_right_num/nmet_total_num)
    print(all_right_num,all_total_num,all_right_num/all_total_num)
                    
if __name__ == '__main__':
    import time
    start=time.time()
    
#    filepath=r'/Volumes/dusten/nc/NUIST.20170524235348.1.nc'
#    radar=core.nc2radar(filepath)
#    
#    metsignal=METSIGNAL(radar)
#    metsignal.feature_calculation()
    
#    sweep=4
#    moment_index=3
#    norm=(0,6)
#    cmap='gist_ncar'
#    cblabel='SD(ZDR) [dB]'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=sweep,moment_index,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    metsignal.feature_visualization(info)
    
#    sweep=4
#    moment_index=4
#    norm=(0,.3)
#    cmap='gist_ncar'
#    cblabel='SD(CC)'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=sweep,moment_index,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    metsignal.feature_visualization(info)
    
#    metsignal.fuzzy_logic()
    
#    sweep=3
#    norm=(0,1)
#    cmap='gist_ncar'
#    cblabel='Aggregation value'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=sweep,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    metsignal.A_visualization(info)
    
#    metsignal.post_processing()
    
#    altitude=3
#    cappi=cappi_calculation(radar,altitude)
#    norm=(-10,60)
#    cmap='gist_ncar'
#    cblabel='Z [dBZ]'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=cappi,altitude,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    cappi_visualization(radar,info)
    
#    metsignal.get_result()
    
#    sweep=1
#    cblabel='MetSignal'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=sweep,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    metsignal.metsignal_visualization(info)
    
#    sweep=4
#    radar.data[(sweep,'Z_MET')]=np.copy(radar.data[(sweep,'Z')])
#    radar.data[(sweep,'Z_MET')][metsignal.result[sweep]==0]=-999
#    moment='Z_MET'
#    norm=(-10,60)
#    cmap='gist_ncar'
#    cblabel='Z_MET [dBZ]'
#    resolution=.075
#    rangecircle=50
#    maxrange=150
#    dpi=300
#    figpath=r'/Users/zhangshuai/Desktop'
#    info=sweep,moment,norm,cmap,cblabel,resolution,rangecircle,maxrange,dpi,figpath
#    core.visualization(radar,info)
    
#    train_table()
#    get_train_set()
#    distribution()
    test(0)
    test(1)
    test(2)
    
    end=time.time()
    print(end-start)