%objective: Assess the variation of wind observation accuracy under different solar background radiation
%when Na reaches the specified value
%date: 20200825

clc;clear;close all
%Define the path and initialization parameters
main_path='E:\data\Aeolus\orbit\Na_SBR_impact\';
atm_path='E:\solar_background_experiments\functions\';
Na_ray=1e3:1e2:1e4;
N_ray=length(Na_ray);
Pek_Na=0.81;
Pek_Nb=0.67;
Nb_ray=Na_ray*Pek_Nb/Pek_Na;
toa_rad=[72.4985,156];
toa_rad=toa_rad*1e6;%Convert into international system units
N_toa=length(toa_rad);

%Interpolate pressure and temperature based on US76 atmosphere
hgh_bin=10;%Assuming that the atmosphere is at 10 km, unit (km)
%Load the temperature and pressure, unit height(km),Pressure(hPa),temperature(K)
load([atm_path,'afglus.mat'],'hgh_atm','Prs_atm','Temp_atm');
prs_bin=interp1(hgh_atm,Prs_atm,hgh_bin);
temp_bin=interp1(hgh_atm,Temp_atm,hgh_bin);

%Find the partial derivative of the LOS wind to the response function at the corresponding temperature and pressure
%Load the database of the partial derivative of the LOS wind to the response function
deri_path='E:\lookup_table\lookup_new_parameter\';
deri_filename='derivate_Wnd2ResFun.mat';
load([deri_path,deri_filename]);
deri_v2r_tmp=interp2(T_deri,P_deri,deri_v2r,...
    temp_bin,prs_bin*0.1);

%% Calculate the number of photons excited by the solar background radiation
%consistant value
c=3e8;     %light speed
h=6.626e-34;%Plank constant
lbd=355e-9;%wavelength of laser
wv=lbd;
E_quan_eff=0.84;%Quantum efficiency of the detector
E_opt_eff=0.34;%efficiency of the optical system
Eo=E_quan_eff*E_opt_eff;
% Ar=15e-6; %solid angle
FOV=18.1e-6;%field of view (FOV),units:(urad)
Ar=2*pi*(1-cos(FOV/2));%convert FOV to solid angle
Phi_R=1.5;   %Telescope diameter
Albd=1e-9;%Interferometer bandwidth
n_pulse=20;%number of laser shots in one measurement
n_mea=30;%number of measurements in one observation

%Calculate the number of photons caused by solar radiation on the Rayleigh channel
photon_ray=zeros(1,N_toa);
delta_R=1e3; %height of range bin
vza_loc=37.6*pi/180;%local observation zenith angle, in radians
Az_ray=delta_R/cos(vza_loc);
for ii_toa=1:N_toa
	photon_ray(ii_toa)=n_mea*n_pulse*Eo*toa_rad(ii_toa)*Ar*(pi*Phi_R^2/4)*Albd*(2*Az_ray/c)/(h*c/lbd);
end

%Simulate the number of photons in the Rayleigh channel and calculate the number of photons generated on the ACCD
%Simulated spectrum, assuming uniform distribution
%parameters of the instrument
FP_fsr=4.56e-12;%FP Free Spectral Rangeunit(m)
wv_step=4e-17;
wv_rag=-FP_fsr/2:wv_step:FP_fsr/2;
N_wv_rag=length(wv_rag);
solar_spm=ones(1,N_wv_rag);
solar_spm=solar_spm/sum(solar_spm);
%the spectrum on Rayleigh channel excited by solar background
N_solar_spm_ray=zeros(N_toa,N_wv_rag);
for ii_toa=1:N_toa
    N_solar_spm_ray(ii_toa,:)=photon_ray(ii_toa)*solar_spm;
end

%Calculate the transmission curve of FP
FP1_PekTrs=Pek_Na;%Peak transmission of FP1
FP2_PekTrs=Pek_Nb;%Peak transmission of FP2
FP_space=2.33e-12;%Filter separation of FPs
FP1_Cen=FP_space/2;
FP2_Cen=-FP_space/2;
FP1_FWHM=0.65155e-12;
FP2_FWHM=0.64315e-12;

%Initialization
FP1_LbdRag_Trs=zeros(1,N_wv_rag);
FP2_LbdRag_Trs=zeros(1,N_wv_rag);
for ii_wv_rag=1:N_wv_rag
    FP1_LbdRag_Trs(ii_wv_rag)=1/(1+((wv_rag(ii_wv_rag)-FP1_Cen)/(FP1_FWHM*0.5))^2);
    FP2_LbdRag_Trs(ii_wv_rag)=1/(1+((wv_rag(ii_wv_rag)-FP2_Cen)/(FP2_FWHM*0.5))^2);
end
FP1_LbdRag_Trs=FP1_LbdRag_Trs/max(FP1_LbdRag_Trs)*FP1_PekTrs;
FP2_LbdRag_Trs=FP2_LbdRag_Trs/max(FP2_LbdRag_Trs)*FP2_PekTrs;

% Calculate the number of photons excited by solar background after passing through the FP interferometers
N_solar_ray1=N_solar_spm_ray*FP1_LbdRag_Trs';
N_solar_ray2=N_solar_spm_ray*FP2_LbdRag_Trs';
N_solar_ray=[N_solar_ray1,N_solar_ray2];

%% Calculate the wind observation uncertainties of Rayleigh channel based on theoretical algorithm
%Initialization
sigma_ResFun_solar=zeros(N_toa,N_ray);
sigma_ray_vlos_solar=zeros(N_toa,N_ray);
for ii_ray=1:N_ray
    for ii_toa=1:N_toa
        Na_tmp=Na_ray(ii_ray);
        Nb_tmp=Nb_ray(ii_ray);
        sigma_a_solar=sqrt(Na_tmp+N_solar_ray(ii_toa,1));
        sigma_b_solar=sqrt(Nb_tmp+N_solar_ray(ii_toa,2));
		%Calculate the uncertainties of response function
        sigma_ResFun_solar(ii_toa,ii_ray)=2/(Na_tmp+Nb_tmp)^2*...
            sqrt(Nb_tmp^2*sigma_a_solar^2+Na_tmp^2*sigma_b_solar^2);
        %wind observation uncertainties based on theoretical algorithm
        sigma_ray_vlos_solar(ii_toa,ii_ray)=abs(deri_v2r_tmp*sigma_ResFun_solar(ii_toa,ii_ray));
    end
end
%Convert wind observation uncertainties of LOS wind to HLOS wind
sigma_ray_hlos_solar=sigma_ray_vlos_solar/sin(vza_loc);

h1=figure(1);
plot(Na_ray,sigma_ray_hlos_solar(1,:))
hold on
plot(Na_ray,sigma_ray_hlos_solar(2,:))
hold off
axis([0,1e4,0,20])
h_legend=legend('typical solar background noise','worst solar background noise');
h_legend.Box='off';
h_legend.Position=[0.156 0.237 0.389 0.09];
xlabel('Useful signal level on channel A (at BRC level)')
ylabel('Rayleigh wind uncertainties, HLOS (m/s)')
